package pl.psnc.expres.broker.remote.experiment;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.expres.broker.config.IBrokerConfiguration;
import pl.psnc.expres.broker.experiment.CorrelationFactory;
import pl.psnc.expres.broker.remote.tn.impl.TranslationJobStub;
import pl.psnc.expres.model.experiment.ExperimentInfo;
import pl.psnc.expres.model.experiment.VlbiExperiment;
import pl.psnc.expres.model.resource.ResourceValue;
import pl.psnc.expres.model.resource.helper.ResourceHelper;
import pl.psnc.expres.model.type.value.FileServerType;
import pl.psnc.expres.remote.vex2ccf.model.CCFValue;
import pl.psnc.vlab.conf.ConfigFactory;

/**
 * {@link HandleExperiment} class - handle new vlbi experiment
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class HandleExperiment {

	/** Stores instance of field: log */
	private final Log log = LogFactory.getLog(HandleExperiment.class);

	/** Stores instance of field: vlbiExperiment */
	private final VlbiExperiment vlbiExperiment;

	/**
	 * Creates a new instance of {@link HandleExperiment} with experiment
	 * details.
	 * 
	 * @param vlbiExperiment experiment details, instance of
	 *        {@link VlbiExperiment}
	 */
	public HandleExperiment(VlbiExperiment vlbiExperiment) {
		CorrelationFactory.newInstance(vlbiExperiment);
		this.vlbiExperiment = vlbiExperiment;
	}

	// -----------------------------------------------------------------------------------
	// ---- Business logic

	/**
	 * Handle new vlbi experiment
	 */
	public void handle() {
		log.debug("<handle>");
		if (vlbiExperiment == null) {
			log.error("</handle: Vlbi experiment is NULL>");
			return;
		}
		ExperimentInfo info = vlbiExperiment.getInfo();
		log.debug("Experiment name=" + info != null ? info.getName() : null);

		List<ResourceValue> resources = vlbiExperiment.getResources();
		List<ResourceValue> fileServerList = ResourceHelper.getConcreteTypes(resources,
				FileServerType.class);
		for (ResourceValue fileServer : fileServerList) {
			startTranslationJob(fileServer);
		}

		log.debug("</handle>");
	}

	// -----------------------------------------------------------------------------------
	// ---- Private helper methods

	/**
	 * Start new translation job
	 * 
	 * @param rValue instance of resource value
	 */
	private void startTranslationJob(ResourceValue rValue) {
		log.debug("<startTranslationJob>");
		if (rValue == null) {
			log.debug("</startTranslationJob: NULL>");
			return;
		}
		log.debug("Starting: " + rValue.getName());
		try {
			//
			IBrokerConfiguration config = (IBrokerConfiguration) ConfigFactory.getConfigFactory();
			CCFValue ccf = vlbiExperiment.getCcf();

			TranslationJobStub.TranslationJobInfo jobInfo = new TranslationJobStub.TranslationJobInfo();
			jobInfo.setBrokerLocation(config.getTnNotificationServiceLocation());
			jobInfo.setChunkSize(1000); // TODO take from WFM
			jobInfo.setStartTime(ccf.getStart());
			jobInfo.setEndTime(ccf.getStop());
			jobInfo.setExperimentName(ccf.getExper_name());
			jobInfo.setGridFtpLocation(rValue.getGridFTPLocation());
			jobInfo.setTelescopeName(rValue.getAbbreviation());

			TranslationJobStub.StartTranslationJob trJob = new TranslationJobStub.StartTranslationJob();
			trJob.setParam0(jobInfo);

			// TODO commented for testing purposes
			// NewTranslationJobStub trStub = new
			// NewTranslationJobStub(rValue.getServiceLocation());
			// trStub.startTranslationJob(trJob);
			log.debug("</startTranslationJob>");
		} catch (Exception e) {
			log.error("</startTranslationJob: " + e.getMessage() + ">");
		}
	}

	// -----------------------------------------------------------------------------------
	// ---- Setters / Getters

	/**
	 * Get value of field: vlbiExperiment
	 * 
	 * @return the vlbiExperiment
	 */
	public VlbiExperiment getVlbiExperiment() {
		return vlbiExperiment;
	}

}
