/**
 * ResourceTypeFactory.java
 * Created on Nov 8, 2007, 3:42:13 PM
 */
package pl.psnc.expres.model.factory;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.expres.model.factory.helper.ModelKeys;
import pl.psnc.expres.model.resource.ResourceMapper;
import pl.psnc.expres.model.resource.ResourceValue;
import pl.psnc.expres.model.util.JsonReaderHelper;
import pl.psnc.vlab.exception.ResourceNotFoundException;

/**
 * {@link ResourceFactory} class - factory is used to get a list of resource.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class ResourceFactory {

	/** Logger instance */
	private static Log log = LogFactory.getLog(ResourceFactory.class);

	/** Stores the list of resource's values */
	private static List<ResourceValue> resources;

	/**
	 * Get the resource by the given id
	 * 
	 * @param id resource id
	 * @return resource value object
	 */
	protected static ResourceValue getResource(int id) throws ResourceNotFoundException {
		return getResource(id, ModelKeys.RESOURCE_LOCATION);
	}

	/**
	 * Get the resource by the given id
	 * 
	 * @param id resource id
	 * @param filePath path to a Json file with resource description
	 * @return resource value object
	 */
	public static ResourceValue getResource(int id, String filePath)
			throws ResourceNotFoundException {
		if (resources == null) {
			log.debug("No resources. Load");
			resources = getResources(filePath);
		}

		for (ResourceValue rValue : resources) {
			log.debug("id=" + rValue.getId());
			if (rValue.getId() == id) {
				return rValue;
			}
		}
		throw new ResourceNotFoundException("Resource [id=" + id + "] has not been found.");
	}

	/**
	 * Get the resource by the given id
	 * 
	 * @param abbreviation resource abbreviation
	 * @param filePath path to a Json file with resource description
	 * @return resource value object
	 */
	public static ResourceValue getResource(String abbreviation, String filePath)
			throws ResourceNotFoundException {
		if (resources == null) {
			log.debug("No resources. Load");
			resources = getResources(filePath);
		}
		for (ResourceValue rValue : resources) {
			String abb = rValue.getAbbreviation();
			if (abb != null && abb.equals(abbreviation)) {
				return rValue;
			}
		}
		throw new ResourceNotFoundException("Resource [abbrevaiotn=" + abbreviation
				+ "] has not been found.");
	}

	/**
	 * Get the resource by the given id
	 * 
	 * @param abbreviation resource abbreviation
	 * @param filePath path to a Json file with resource description
	 * @return resource value object
	 */
	public static ResourceValue getResource(String abbreviation, InputStream resourceIn)
			throws ResourceNotFoundException {
		if (resources == null) {
			log.debug("No resources. Load");
			resources = getResources(resourceIn);
		}
		for (ResourceValue rValue : resources) {
			String abb = rValue.getAbbreviation();
			if (abb != null && abb.equals(abbreviation)) {
				return rValue;
			}
		}
		throw new ResourceNotFoundException("Resource [abbrevaiotn=" + abbreviation
				+ "] has not been found.");
	}

	/**
	 * Get the resource list
	 * 
	 * @param filePath path to a Json file with resource description
	 * @return resource ResourceValues array
	 */
	public static List<ResourceValue> getResources(String filePath)
			throws ResourceNotFoundException {
		try {
			InputStream inStream = new FileInputStream(new File(filePath));
			return getResources(inStream);
		} catch (Exception e) {
			throw new ResourceNotFoundException(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * Get the resource list from string
	 * 
	 * @param resourceAsString resource text representation
	 * @return resource ResourceValues array
	 */
	public static List<ResourceValue> getResourcesFromString(String resourceAsString)
			throws ResourceNotFoundException {
		try {
			InputStream inStream = new ByteArrayInputStream(resourceAsString.getBytes());
			return getResources(inStream);
		} catch (Exception e) {
			throw new ResourceNotFoundException(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * Get the resource list
	 * 
	 * @param inStream input stream with resource description (Json format)
	 * @return resource ResourceValues array
	 */
	public static List<ResourceValue> getResources(InputStream inStream)
			throws ResourceNotFoundException {
		if (resources == null) {
			Object res = JsonReaderHelper.load(inStream, ResourceMapper.class);
			if (res == null) {
				log.error("ResourceValues are NULL");
				return null;
			}
			resources = ((ResourceMapper) res).getResources();
		}
		return resources;
	}

	/**
	 * Get the resource list from default location defined by
	 * ModelKeys.RESOURCE_LOCATION
	 * 
	 * @param filePath path to a Json file with resource description
	 * @return resource ResourceValues array
	 */
	public static List<ResourceValue> getResources() throws ResourceNotFoundException {
		if (resources == null) {
			return getResources(ModelKeys.RESOURCE_LOCATION);
		}
		return resources;
	}

}
