package pl.psnc.graph.io;

import java.awt.Component;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.prefs.Preferences;

import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileSystemView;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.graph.i18n.GraphBundleFactory;
import pl.psnc.graph.i18n.helper.GraphKeys;
import pl.psnc.vlab.util.FileTools;
import pl.psnc.vlab.util.gui.message.JMessage;

/**
 * {@link GraphFileChooser} class - graph file chooser class.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class GraphFileChooser extends JFileChooser {

	/** Default serial uid */
	private static final long serialVersionUID = 1L;

	/** Instance of logger */
	private Log log = LogFactory.getLog(this.getClass().getName());

	/** Stores instance of field: LAST_PATH */
	public static final String LAST_PATH = "LastPathPosition";

	/** Stores instance of field: LAST_FILE_FILTER */
	public static final String LAST_FILE_FILTER = "LastFileFilter";

	/** Stores instance of JDialog */
	private JDialog dialog;

	/** Stores instance of field: returnValue */
	private int returnValue = ERROR_OPTION;

	/**
	 * Constructor for DynScenFileChooser.
	 */
	public GraphFileChooser() {
	}

	/**
	 * Creates a new File Chooser with the list of specified file filters.
	 * 
	 * @param fileFilterList list of file filters
	 */
	public GraphFileChooser(FileFilterType... fileFilterList) {
		this();
		if (fileFilterList == null) {
			return;
		}
		for (FileFilterType fileFilter : fileFilterList) {
			addChoosableFileFilter(fileFilter.getFileFilter());
		}
	}

	/**
	 * Constructor for DynScenFileChooser.
	 * 
	 * @param currentDirectoryPath
	 */
	public GraphFileChooser(String currentDirectoryPath) {
		super(currentDirectoryPath);
		init();
	}

	/**
	 * Constructor for DynScenFileChooser.
	 * 
	 * @param currentDirectory
	 */
	public GraphFileChooser(File currentDirectory) {
		super(currentDirectory);
		init();
	}

	/**
	 * Constructor for DynScenFileChooser.
	 * 
	 * @param fsv
	 */
	public GraphFileChooser(FileSystemView fsv) {
		super(fsv);
		init();
	}

	/**
	 * Constructor for DynScenFileChooser.
	 * 
	 * @param currentDirectory
	 * @param fsv
	 */
	public GraphFileChooser(File currentDirectory, FileSystemView fsv) {
		super(currentDirectory, fsv);
		init();
	}

	/**
	 * Constructor for DynScenFileChooser.
	 * 
	 * @param currentDirectoryPath
	 * @param fsv
	 */
	public GraphFileChooser(String currentDirectoryPath, FileSystemView fsv) {
		super(currentDirectoryPath, fsv);
		init();
	}

	/**
	 * Initialises the gui.
	 * 
	 * Adds a property change listener to catch changes at the file filter. If
	 * the user changes the file filter the listener changes the accessory for
	 * the correct file format.
	 */
	protected void init() {

		// if the dialog was committed we
		// store the path position for the
		// next call
		addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (e.getActionCommand() == JFileChooser.APPROVE_SELECTION) {
					try {
						Preferences.userNodeForPackage(this.getClass()).put(LAST_PATH,
								GraphFileChooser.this.getSelectedFile().getPath());
					} catch (Exception ex) {
						// ignore
					}
				}
			}
		});

		/*
		 * addPropertyChangeListener(new PropertyChangeListener() { /**
		 * 
		 * @see java.beans.PropertyChangeListener#propertyChange(PropertyChangeEvent)
		 * 
		 * public void propertyChange(PropertyChangeEvent evt) { if (evt != null &&
		 * evt.getPropertyName() != null && (evt.getPropertyName().equals(
		 * JFileChooser.FILE_FILTER_CHANGED_PROPERTY) || evt
		 * .getPropertyName().equals(
		 * JFileChooser.DIALOG_TYPE_CHANGED_PROPERTY))) { FileFilter ff =
		 * (FileFilter) getFileFilter(); GraphModelFileFormat gmff =
		 * GraphModelProviderRegistry .getGraphModelFileFormat(ff); if (gmff ==
		 * null) setAccessory(null); else if (getDialogType() == SAVE_DIALOG)
		 * setAccessory(gmff.getWriteAccessory()); else if (getDialogType() ==
		 * OPEN_DIALOG) setAccessory(gmff.getReadAccessory()); Component comp =
		 * GraphFileChooser.this;
		 * 
		 * while (!(comp instanceof Window)) { if (comp == null) return; comp =
		 * comp.getParent(); } if (comp != null) ((Window) comp).pack(); } } });
		 */

		try {

			String fileName = Preferences.userNodeForPackage(this.getClass()).get(LAST_PATH,
					System.getProperty("user.home"));
			try {
				setCurrentDirectory(new File(fileName));
			} catch (Exception ex) {
				// ignore
			}

		} catch (Exception ex) {
			// ignore
		}

	}

	/**
	 * Called by the UI when the user hits the Approve button (labeled "Open" or
	 * "Save", by default). This can also be called by the programmer. This
	 * method causes an action event to fire with the command string equal to
	 * <code>APPROVE_SELECTION</code>.
	 * 
	 * @see #APPROVE_SELECTION
	 */
	public void approveSelection() {
		try {
			log.debug("<approveSelection>");
			returnValue = APPROVE_OPTION;
			File file = getSelectedFile();
			log.debug("File=" + file);
			// check whether the file exists
			if (getDialogType() == SAVE_DIALOG && FileTools.fileExists(file)) {
				log.debug("save");
				String msg = GraphBundleFactory.getValue(GraphKeys.FILE_EXISTS);
				int res = JMessage.showConfirmQuestionMessage(this, msg, JMessage.YES_NO_OPTION);
				if (res == JMessage.NO_OPTION) {
					fireActionPerformed(APPROVE_SELECTION);
					return;
				}
			} // end of file exists

			if (dialog != null) {
				dialog.setVisible(false);
			}
			fireActionPerformed(APPROVE_SELECTION);
			log.debug("</approveSelection>");
		} catch (Exception e) {
			log.error(e.getMessage());
		}
	}

	/**
	 * Called by the UI when the user chooses the Cancel button. This can also
	 * be called by the programmer. This method causes an action event to fire
	 * with the command string equal to <code>CANCEL_SELECTION</code>.
	 * 
	 * @see #CANCEL_SELECTION
	 */
	public void cancelSelection() {
		returnValue = CANCEL_OPTION;
		if (dialog != null) {
			dialog.setVisible(false);
		}
		fireActionPerformed(CANCEL_SELECTION);
	}

	/**
	 * Pops a custom file chooser dialog with a custom approve button. For
	 * example, the following code pops up a file chooser with a "Run
	 * Application" button (instead of the normal "Save" or "Open" button):
	 * 
	 * <pre>
	 * filechooser.showDialog(parentFrame, &quot;Run Application&quot;);
	 * </pre>
	 * 
	 * Alternatively, the following code does the same thing:
	 * 
	 * <pre>
	 * JFileChooser chooser = new JFileChooser(null);
	 * chooser.setApproveButtonText(&quot;Run Application&quot;);
	 * chooser.showDialog(parentFrame, null);
	 * </pre>
	 * 
	 * <!--PENDING(jeff) - the following method should be added to the api:
	 * showDialog(Component parent);--> <!--PENDING(kwalrath) - should specify
	 * modality and what "depends" means.-->
	 * 
	 * <p>
	 * 
	 * The <code>parent</code> argument determines two things: the frame on
	 * which the open dialog depends and the component whose position the look
	 * and feel should consider when placing the dialog. If the parent is a
	 * <code>Frame</code> object (such as a <code>JFrame</code>) then the
	 * dialog depends on the frame and the look and feel positions the dialog
	 * relative to the frame (for example, centered over the frame). If the
	 * parent is a component, then the dialog depends on the frame containing
	 * the component, and is positioned relative to the component (for example,
	 * centered over the component). If the parent is <code>null</code>, then
	 * the dialog depends on no visible window, and it's placed in a
	 * look-and-feel-dependent position such as the center of the screen.
	 * 
	 * @param parent the parent component of the dialog; can be
	 *        <code>null</code>
	 * @param approveButtonText the text of the <code>ApproveButton</code>
	 * @return the return state of the file chooser on popdown:
	 *         <ul>
	 *         <li>JFileChooser.CANCEL_OPTION
	 *         <li>JFileChooser.APPROVE_OPTION
	 *         <li>JFileCHooser.ERROR_OPTION if an error occurs or the dialog
	 *         is dismissed
	 *         </ul>
	 * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
	 *            true.
	 * @see java.awt.GraphicsEnvironment#isHeadless
	 */
	public int showDialog(Component parent, String approveButtonText)
			throws java.awt.HeadlessException {

		if (approveButtonText != null) {
			setApproveButtonText(approveButtonText);
			setDialogType(CUSTOM_DIALOG);
		}
		dialog = createDialog(parent);
		dialog.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				returnValue = CANCEL_OPTION;
			}
		});
		returnValue = ERROR_OPTION;
		rescanCurrentDirectory();

		dialog.setVisible(true);
		// dialog.dispose();
		// dialog = null;
		return returnValue;
	}

}
