/*
 * JBallsPane.java
 *
 * Created on March 20, 2006, 5:08 PM
 */

package pl.psnc.vlab.util.swing.gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.awt.image.IndexColorModel;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.util.Locale;

import javax.swing.JFrame;
import javax.swing.JPanel;

import pl.psnc.vlab.exception.VlabException;
import pl.psnc.vlab.util.i18n.ResourceBundleManager;
import pl.psnc.vlab.util.swing.thread.ProgressDialog;

/**
 * {@link JBallsPane} class - pane with flying balls.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
@SuppressWarnings("deprecation")
public class JBallsPane extends JPanel implements Runnable, ProgressDialog {

	/** Stores instance of field: serialVersionUID */
	private static final long serialVersionUID = 1L;

	/**
	 * Stores the ballsArray colors
	 */
	private static Color colors[] = { Color.red, Color.orange, Color.green, Color.blue, Color.pink,
			Color.magenta, Color.black, Color.cyan };

	/** Stores instance of field: thread */
	private Thread thread;

	/** Stores instance of field: bimg */
	private BufferedImage bimg;

	/** Stores instance of field: now */
	private long now, deltaT, lasttime;

	/** Stores instance of field: active */
	private boolean active;

	/** Stores wait text, which will be displayed on the pane */
	private static String waitText;

	/** Stores the ballsArray array */
	protected Ball ballsArray[];

	/** Stores dots */
	private String DOT = " ";

	/** Stores the last dot occurrence */
	private long last;

	/** Delay between dot occurrence */
	private static final int DELAY = 1000 * 3;

	/** Border size */
	private static final int BORDER_SIZE = 2;

	/** Stores instance of field: gradient */
	private static GradientPaint gradient;

	/** Stores instance of field: dialog */
	private JFrame dialog;

	/**
	 * Creates new form Balls with the number of balls and random flag
	 * specified.
	 * 
	 * @param locale current locale
	 * @param ballNumber specifies the number of balls to create
	 * @param randomOccurance If random flag is true the ball occurrence will be
	 *        based on random function. If the flag is false all the balls are
	 *        visible.
	 */
	public JBallsPane(int ballNumber, boolean randomOccurance, Locale locale) throws VlabException {
		try {
			initComponents();
			Dimension size = getPreferredSize();
			gradient = new GradientPaint(0, 0, ColorKeys.GRADIENT_LIGHT_GRAY_AND_WHITE[1],
					size.width - BORDER_SIZE, size.height - BORDER_SIZE,
					ColorKeys.GRADIENT_LIGHT_GRAY_AND_WHITE[0]);

			ballsArray = new Ball[ballNumber];
			for (int i = 0; i < ballNumber; i++) {
				ballsArray[i] = new Ball(colors[i % colors.length], 15);
				if (randomOccurance) {
					ballsArray[i].isSelected = Math.random() > 0.5;
				} else {
					ballsArray[i].isSelected = true;
				}
			} // end of for
			ResourceBundleManager bundle = new ResourceBundleManager(
					"pl.psnc.vlab.util.swing.gui.JBallsPane", locale == null ? Locale.getDefault()
							: locale);
			waitText = bundle.getValue("wait.message");
		} catch (Exception e) {
			throw new VlabException("Error during form initialization.Details:" + e.getMessage(), e);
		}
	}

	/**
	 * This method is called from within the constructor to initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is always
	 * regenerated by the Form Editor.
	 */
	// <editor-fold defaultstate="collapsed" desc=" Generated Code
	// ">//GEN-BEGIN:initComponents
	private void initComponents() {

		setLayout(new java.awt.BorderLayout(10, 10));

		setBackground(new java.awt.Color(0, 0, 0));
		setMaximumSize(new java.awt.Dimension(200, 150));
		setMinimumSize(new java.awt.Dimension(200, 150));
		setPreferredSize(new java.awt.Dimension(200, 150));
	}// </editor-fold>//GEN-END:initComponents

	// Variables declaration - do not modify//GEN-BEGIN:variables
	// End of variables declaration//GEN-END:variables

	/**
	 * Defines the ball step
	 * 
	 * @param w
	 * @param h
	 */
	public void step(int w, int h) {
		if (lasttime == 0) {
			lasttime = System.currentTimeMillis();
		}
		now = System.currentTimeMillis();
		deltaT = now - lasttime;
		// deltaT = (now - lasttime) + (long) (5.0 *Math.random());
		// deltaT += (long) 13.0 * Math.random();
		active = false;
		for (int i = 0; i < ballsArray.length; i++) {
			if (ballsArray[i] == null) {
				return;
			}

			// ballsArray[i].Vx += (ballsArray[i].Vx * (float) Math.random()) ;

			ballsArray[i].step(deltaT, w, h);
			if (ballsArray[i].Vy > .02 || -ballsArray[i].Vy > .02
					|| ballsArray[i].y + ballsArray[i].bsize < h) {
				active = true;
			}
		}
		if (!active) {
			for (int i = 0; i < ballsArray.length; i++) {
				ballsArray[i].Vx = ((float) Math.random() / 4.0f - 0.125f); // +
				// (float)
				// (13.0
				// *Math.random())
				// ;
				ballsArray[i].Vy = -(float) Math.random() / 4.0f - 0.2f;
			}
		}
	}

	public void drawDemo(int w, int h, Graphics2D g2) {
		for (int i = 0; i < ballsArray.length; i++) {
			Ball b = ballsArray[i];
			if (b == null || b.imgs[b.index] == null || !b.isSelected) {
				continue;
			}
			g2.drawImage(b.imgs[b.index], (int) b.x, (int) b.y, this);
		}
		lasttime = now;
	}

	/**
	 * Creates a new instance of {@link Graphics2D} object
	 * 
	 * @param w width
	 * @param h height
	 * @return new instance of {@link Graphics2D} object
	 */
	public Graphics2D createGraphics2D(int w, int h) {
		Graphics2D g2 = null;
		if (bimg == null || bimg.getWidth() != w || bimg.getHeight() != h) {
			bimg = (BufferedImage) createImage(w, h);
		}
		g2 = bimg.createGraphics();
		g2.setBackground(getBackground());
		g2.clearRect(0, 0, w, h);

		return g2;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.JComponent#paint(java.awt.Graphics)
	 */
	@SuppressWarnings("unused")
	public void paint(Graphics g) {

		if (last == 0) {
			last = System.currentTimeMillis();
		}
		Dimension d = getSize();
		step(d.width, d.height);
		Graphics2D g2 = createGraphics2D(d.width, d.height);
		// draw gradient
		g2.setPaint(gradient);
		g2.fillRect(0, 0, d.width - BORDER_SIZE, d.height - BORDER_SIZE);

		drawDemo(d.width, d.height, g2);

		// draw wait text
		g2.setColor(Color.BLACK);
		g2.setFont(new Font("Monospaced", Font.BOLD, 14));
		FontMetrics metrics = g2.getFontMetrics();
		int fWidth = metrics.stringWidth(waitText);

		int fHeight = metrics.getHeight();
		g2.drawString(waitText, d.width / 2 - fWidth / 2, d.height - d.height / 3);
		// draw dots
		if (System.currentTimeMillis() - last >= DELAY) {
			DOT = DOT.length() <= 3 ? DOT += "." : " ";
			last = System.currentTimeMillis();
		}
		g2.drawString(DOT, d.width / 2 - fWidth / 2 + fWidth, d.height - d.height / 3);
		g2.dispose();
		g.drawImage(bimg, 0, 0, this);
	}

	public void start() {
		thread = new Thread(this);
		thread.setPriority(Thread.MIN_PRIORITY);
		thread.start();
	}

	public synchronized void stop() {
		thread = null;
	}

	public void run() {
		Thread me = Thread.currentThread();
		while (thread == me) {
			repaint();
			try {
				Thread.sleep(10);
			} catch (InterruptedException e) {
				break;
			}
		}
		thread = null;
	}

	/**
	 * Represents a ball class
	 */
	@SuppressWarnings("unused")
	static class Ball {

		public int bsize;
		public float x, y;
		public float Vx = 0.1f;
		public float Vy = 0.05f;
		public int nImgs = 5;
		public BufferedImage imgs[];
		public int index = (int) (Math.random() * (nImgs - 1));

		private final float inelasticity = .96f;
		private final float Ax = 0.0f;
		private final float Ay = 0.0002f;

		private final float Ar = 0.9f;
		private final int UP = 0;
		private final int DOWN = 1;
		private int indexDirection = UP;
		private boolean collision_x, collision_y;
		private float jitter;
		private Color color;
		private boolean isSelected;

		public Ball(Color color, int bsize) {
			this.color = color;
			makeImages(bsize);
		}

		public void makeImages(int bsize) {
			this.bsize = bsize * 2;
			int R = bsize;
			byte[] data = new byte[R * 2 * R * 2];
			int maxr = 0;
			for (int Y = 2 * R; --Y >= 0;) {
				int x0 = (int) (Math.sqrt(R * R - (Y - R) * (Y - R)) + 0.5);
				int p = Y * (R * 2) + R - x0;
				for (int X = -x0; X < x0; X++) {
					int x = X + 15;
					int y = Y - R + 15;
					int r = (int) (Math.sqrt(x * x + y * y) + 0.5);
					if (r > maxr) {
						maxr = r;
					}
					data[p++] = r <= 0 ? 1 : (byte) r;
				}
			}

			imgs = new BufferedImage[nImgs];

			int bg = 255;
			byte red[] = new byte[256];
			red[0] = (byte) bg;
			byte green[] = new byte[256];
			green[0] = (byte) bg;
			byte blue[] = new byte[256];
			blue[0] = (byte) bg;

			// for each image, set its color
			for (int r = 0; r < imgs.length; r++) {
				float b = 0.5f + (float) ((r + 1f) / imgs.length / 2f);
				for (int i = maxr; i >= 1; --i) {
					float d = (float) i / maxr;
					red[i] = (byte) blend(blend(color.getRed(), 255, d), bg, b);
					green[i] = (byte) blend(blend(color.getGreen(), 255, d), bg, b);
					blue[i] = (byte) blend(blend(color.getBlue(), 255, d), bg, b);
				}
				IndexColorModel icm = new IndexColorModel(8, maxr + 1, red, green, blue, 0);
				DataBufferByte dbb = new DataBufferByte(data, data.length);
				int bandOffsets[] = { 0 };
				WritableRaster wr = Raster.createInterleavedRaster(dbb, R * 2, R * 2, R * 2, 1,
						bandOffsets, null);
				imgs[r] = new BufferedImage(icm, wr, icm.isAlphaPremultiplied(), null);
			}
		}

		// performs the blending of white and color
		private final int blend(int fg, int bg, float fgfactor) {
			return (int) (bg + (fg - bg) * fgfactor);
		}

		// performs the action of bouncing off the window "walls"
		public void step(long deltaT, int w, int h) {
			collision_x = false;
			collision_y = false;

			jitter = (float) Math.random() * .01f - .005f;

			x += Vx * deltaT + (Ax / 2.0) * deltaT * deltaT;
			y += Vy * deltaT + (Ay / 2.0) * deltaT * deltaT;
			if (x <= 0.0f) {
				x = 0.0f;
				Vx = -Vx * inelasticity + jitter;
				collision_x = true;
			}
			if (x + bsize >= w) {
				x = w - bsize;
				Vx = -Vx * inelasticity + jitter;
				collision_x = true;
			}
			if (y <= 0) {
				y = 0;
				Vy = -Vy * inelasticity + jitter;
				collision_y = true;
			}
			if (y + bsize >= h) {
				y = h - bsize;
				Vx *= inelasticity;
				Vy = -Vy * inelasticity + jitter;
				collision_y = true;
			}
			Vy = Vy + Ay * deltaT;
			Vx = Vx + Ax * deltaT;

			if (indexDirection == UP) {
				index++;
			}
			if (indexDirection == DOWN) {
				--index;
			}
			if (index + 1 == nImgs) {
				indexDirection = DOWN;
			}
			if (index == 0) {
				indexDirection = UP;
			}
		}
	} // End Ball class

	public static void main(String argv[]) {
		try {
			final JBallsPane demo = new JBallsPane(4, false, null);
			demo.setVisible(true);
			demo.start();
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * Makes the component visible or invisible. Overrides
	 * <code>Component.setVisible</code>.
	 * 
	 * 
	 * @param aFlag true to make the component visible; false to make it
	 *        invisible
	 */
	public void setVisible(boolean aFlag) {
		super.setVisible(aFlag);
		if (aFlag) {
			dialog = dialog == null ? new JFrame(waitText) : dialog;
			dialog.getContentPane().add(this);
			dialog.setUndecorated(true);
			dialog.pack();
			// center
			GuiTools.center(dialog);
			dialog.setVisible(true);
		} else {
			dialog.setVisible(false);
		}
	}

}
