/*
 * ResourceBundleManager.java
 *
 * Created on 13 luty 2004, 16:06
 */

package pl.psnc.vlab.util.i18n;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * {@link ResourceBundleManager} class - used for formating compound messages.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class ResourceBundleManager {

	/** Instance of message bundle */
	private ResourceBundle bundle;

	/** Current locale */
	private Locale locale;

	/** Value key for not found message */
	private String valueNotFound;

	/**
	 * Creates a new instance of ResourceBundleManager
	 * 
	 * @param bundle resource bundle
	 * @throws MissingResourceException exception which is thrown when no
	 *             resource bundle is found
	 */
	public ResourceBundleManager(String bundle) throws MissingResourceException {
		this(bundle, null);
	}

	/**
	 * Creates a new instance of ResourceBundleManager
	 * 
	 * @param bundle resource bundle
	 * @param locale current locale
	 * @throws MissingResourceException exception which is thrown when no
	 *             resource bundle is found
	 */
	public ResourceBundleManager(String bundle, Locale locale) throws MissingResourceException {
		this(bundle, locale, null);
	}

	/**
	 * Creates a new instance of ResourceBundleManager
	 * 
	 * @param notFoundKey key which will be used when searched key is not found
	 * @param bundle resource bundle
	 * @param locale current locale
	 * @throws MissingResourceException exception which is thrown when no
	 *             resource bundle is found
	 */
	public ResourceBundleManager(String bundle, Locale locale, String notFoundKey)
			throws MissingResourceException {
		this.locale = locale != null ? locale : Locale.getDefault();
		this.bundle = ResourceBundle.getBundle(bundle, this.locale);
		try {
			this.valueNotFound = this.bundle.getString(notFoundKey);
		} catch (Exception e) {
			// ok to ignore exception
			this.valueNotFound = "Invalid ResourceBundle. Couldn't find notFoundKey resource";
		}
	}

	/**
	 * Get the formatted key
	 * 
	 * @param template message template
	 * @param messageArguments message arguments
	 * @return value for the given key
	 */
	public String getFormattedValue(String template, Object[] messageArguments) {
		MessageFormat formatter = new MessageFormat("");
		formatter.setLocale(this.locale);
		formatter.applyPattern(this.bundle.getString(template));
		return formatter.format(messageArguments);

	}

	/**
	 * Get the value for the given key for corresponding resource bundle
	 * 
	 * @return value for the given key
	 * @param key key for the desired string
	 */
	public String getValue(String key) {
		try {
			return bundle.getString(key);
		} catch (MissingResourceException missExc) {
			return valueNotFound;
		} catch (Exception e) {
			return "Error while reading resource key: " + key + ". Details:" + e.getMessage();
		}
	}

	/**
	 * Test whether the given key is defined in the current bundle
	 * 
	 * @return <code>true</code> if key was found, <code>false</code>
	 *         otherwise
	 * @param key key for the desired string
	 */
	public Boolean hasKey(String key) {
		try {
			bundle.getString(key);
			return Boolean.TRUE;
		} catch (Exception e) {
			return Boolean.FALSE;
		}
	}

}