/**
 * 
 */
package pl.psnc.vlab.util.gui.model.info;

import javax.swing.JTable;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * {@link ModelProperties} class - used to store model properties like column
 * names, editable columns, column types
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class ModelProperties {

	/** Stores Logger instance */
	private final Log log = LogFactory.getLog(ModelProperties.class);

	/** Stores a list of column types */
	private ColumnInfo[] columns;

	/** Specifies editable columns */
	private boolean[] editable;

	/** Stores sort order for the given model */
	private ColumnInfo[] sortOrder;

	/**
	 * Stores selected property - tells whether model values are selected by
	 * default
	 */
	boolean selected;

	// -------------------------------------------------------------------------
	// ---- Public Helper methods

	/**
	 * Tells whether the given column can be edited
	 * 
	 * @param columnIndex column index
	 * @return <code>true</code> if column is editable, <code>false</code>
	 *         otherwise
	 */
	public boolean isEditable(int columnIndex) {
		int size = editable != null ? editable.length : 0;
		if (columnIndex >= size) {
			return false;
		}
		return this.editable[columnIndex];
	}

	/**
	 * Decorated JTable Columns
	 * 
	 * @param table
	 */
	public void decorateColums(JTable table) {
		log.debug("<decorateColums>");
		if (columns == null) {
			log.warn("Colums are NULL");
			return;
		}
		TableColumnModel colModel = table != null ? table.getColumnModel() : null;
		if (colModel == null) {
			log.warn("Table model was not found. NULL");
			return;
		}
		for (ColumnInfo type : columns) {
			decorate(type, colModel);
		}
	}

	/**
	 * Get index for the given column type
	 * 
	 * @param type column type
	 * @return index of column type
	 */
	public Integer getColumnIndex(ColumnInfo type) {
		log.debug("<getColumnIndex: type=" + type + ">");
		int size = columns != null ? columns.length : 0;
		if (type == null || size <= 0) {
			log.warn("Bad type or no colum types defined");
			return null;
		}
		for (int i = 0; i < size; i++) {
			if (columns[i].equals(type)) {
				log.debug("</getColumnIndex: " + i + ">");
				return i;
			}
		}
		log.debug("<getColumnIndex: NOT FOUND>");
		return null;
	}

	/**
	 * Get Column for the given column index
	 * 
	 * @param columnIndex column index
	 * @return ColumnType for the given column index
	 */
	public ColumnInfo getColumnType(int columnIndex) {
		int size = columns != null ? columns.length : 0;
		if (size < columnIndex) {
			log.error("<getColumnType: bad index=" + columnIndex + ">");
			return null;
		}
		return columns[columnIndex];
	}

	/**
	 * Get list of column names
	 * 
	 * @return list of column names
	 */
	public String[] getColumnNames() {
		int size = columns != null ? columns.length : 0;
		String result[] = new String[size];
		for (int i = 0; i < size; i++) {
			result[i] = columns[i].getColumnType().getName();
		}
		return result;
	}

	/**
	 * Get list of columns types
	 * 
	 * @return list of columns types
	 */
	public Class<?>[] getColumnTypes() {
		int size = columns != null ? columns.length : 0;
		Class<?> result[] = new Class<?>[size];
		for (int i = 0; i < size; i++) {
			result[i] = columns[i].getColumnType().getType();
		}
		return result;
	}

	// -------------------------------------------------------------------------
	// ---- Private Helper methods

	/**
	 * Set size of the given column
	 * 
	 * @param type
	 * @param colModel
	 */
	private void decorate(ColumnInfo type, TableColumnModel colModel) {
		ColumnDecorator decorator = type.getColumnType().getDecorator();
		if (decorator == null) {
			return;
		}

		Integer index = getColumnIndex(type);
		log.debug("Decorating: " + type + " index=" + index);
		if (index == null) {
			log.warn("Wrong index value -> NULL ");
			return;
		}
		TableColumn tableColumn = colModel.getColumn(index);
		Integer value = null;
		if ((value = decorator.getPrefered()) != null) {
			log.debug("Setting pref=" + value);
			tableColumn.setPreferredWidth(value);
		}
		if ((value = decorator.getMin()) != null) {
			tableColumn.setMinWidth(value);
		}
		if ((value = decorator.getMax()) != null) {
			tableColumn.setMaxWidth(value);
		}
	}

	// -------------------------------------------------------------------------
	// ---- Setters / Getters

	/**
	 * Get value of field: columns
	 * 
	 * @return the columns
	 */
	public ColumnInfo[] getColumns() {
		return columns;
	}

	/**
	 * Set value of field: columns
	 * 
	 * @param columns the columns to set
	 */
	public void setColumns(ColumnInfo[] columns) {
		this.columns = columns;
	}

	/**
	 * Get value of field: editable
	 * 
	 * @return the editable
	 */
	public boolean[] getEditable() {
		return editable;
	}

	/**
	 * Set value of field: editable
	 * 
	 * @param editable the editable to set
	 */
	public void setEditable(boolean[] editable) {
		this.editable = editable;
	}

	/**
	 * Get value of field: sortOrder
	 * 
	 * @return the sortOrder
	 */
	public ColumnInfo[] getSortOrder() {
		return sortOrder;
	}

	/**
	 * Set value of field: sortOrder
	 * 
	 * @param sortOrder the sortOrder to set
	 */
	public void setSortOrder(ColumnInfo[] sortOrder) {
		this.sortOrder = sortOrder;
	}

	/**
	 * Get value of field: selected
	 * 
	 * @return the selected
	 */
	public boolean isSelected() {
		return selected;
	}

	/**
	 * Set value of field: selected
	 * 
	 * @param selected the selected to set
	 */
	public void setSelected(boolean selected) {
		this.selected = selected;
	}

}
