package pl.psnc.vlab.util.gui.message;

import java.awt.Component;
import java.util.Locale;
import java.util.MissingResourceException;

import javax.swing.JOptionPane;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.i18n.ResourceBundleManager;

/**
 * {@link JMessage} class - abstract class defines a set of message dialogs
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class JMessage extends JOptionPane {

	/** Stores instance of field: log */
	private final static Log log = LogFactory.getLog(JMessage.class);

	/** Holds a path to i18n resource file */
	private static final String RESOURCE_BUNDLE = "bundle/JMessage";

	/** Holds instance of Resource Bundle Manager */
	private static ResourceBundleManager bundleManager;

	/**
	 * Shows JDialog Information Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * 
	 */
	public static void showInfoMessage(Component object, Object message) {
		showMessage(object, message, INFORMATION_MESSAGE);
	}

	/**
	 * Shows JDialog Error Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * 
	 */
	public static void showErrorMessage(Component object, Object message) {
		showMessage(object, message, ERROR_MESSAGE);
	}

	/**
	 * Shows JDialog Error Message. Based on the exception type and error code
	 * different errors icons are displayed.
	 * 
	 * @param object the parent component
	 * @param error instance of error
	 * 
	 */
	public static void showErrorMessage(Component object, Exception error) {
		String message = error != null ? error.getLocalizedMessage()
				: "Error occured. Please contact system administrator";		

		log.error(String.format("Error message=%s", message));
		showMessage(object, message, ERROR_MESSAGE);
	}

	/**
	 * Shows JDialog Question Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * 
	 */
	public static void showQuestionMessage(Component object, Object message) {
		showMessage(object, message, QUESTION_MESSAGE);
	}

	/**
	 * Shows JDialog Warning Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * 
	 */
	public static void showWarningMessage(Component object, Object message) {
		showMessage(object, message, WARNING_MESSAGE);
	}

	/**
	 * Shows JDialog Confirm Question Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * @param optionType option type
	 * 
	 */
	public static int showConfirmQuestionMessage(Component object, Object message, int optionType) {
		return showConfirmDialog(object, message, getWindowTitle(QUESTION_MESSAGE), optionType,
				QUESTION_MESSAGE);
	}

	/**
	 * Shows JDialog Confirm Question Message
	 * 
	 * @param object the parent component
	 * @param message message
	 * @param optionType option type
	 * 
	 */
	public static int showConfirmInfoMessage(Component object, Object message, int optionType,
			Object options[]) {
		return showOptionDialog(object, message, getWindowTitle(INFORMATION_MESSAGE), optionType,
				INFORMATION_MESSAGE, null, options, null);
	}

	// ---------------------------------------------------------------------
	// ---- Private helper methods

	/**
	 * Show JOption Dialog message
	 * 
	 * @param object the parent component
	 * @param message message
	 * @param type dialog type: info, error, warning, question (based on
	 *        JOptionPane message types)
	 */
	private static void showMessage(Component object, Object message, int type) {
		JOptionPane.showMessageDialog(object, message, getWindowTitle(type), type);
	}

	/**
	 * Get JOption Dialog title. The tile is taken from resource bundle :
	 * RESOURCE_BUNDLE
	 * 
	 * @param type type of the given dialog
	 * @return return dialog window title
	 */
	private static String getWindowTitle(int type) {
		return getWindowTitle(type, false);
	}

	/**
	 * Get JOption Dialog title. The tile is taken from resource bundle :
	 * RESOURCE_BUNDLE
	 * 
	 * @param type type of the given dialog
	 * @param secondError specifies whether return second error title
	 * @return return dialog window title
	 */
	private static String getWindowTitle(int type, boolean secondError) {
		if (bundleManager == null) {
			try {
				bundleManager = new ResourceBundleManager(RESOURCE_BUNDLE, Locale.getDefault());
			} catch (MissingResourceException e) {
				LogFactory.getLog(JMessage.class.getName()).error(
						"Error while loading resource file: " + RESOURCE_BUNDLE + ". Details:"
								+ e.getMessage());
				return "";
			}
		}

		switch (type) {
		case INFORMATION_MESSAGE:
			return bundleManager.getValue("form.title.info");

		case WARNING_MESSAGE:
			return bundleManager.getValue("form.title.warning");

		case QUESTION_MESSAGE:
			return bundleManager.getValue("form.title.question");

		case ERROR_MESSAGE:
			if (secondError) {
				return bundleManager.getValue("form.title.second.error");
			}
			return bundleManager.getValue("form.title.error");

		}
		return "";
	}

}