package pl.psnc.vlab.util.gui.helper;

import javax.swing.JTable;

/**
 * {@link TableHelper} class - set of utility classes for JTable.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class TableHelper {

	/**
	 * Maps the index of the row in terms of the view to the underlying
	 * TableModel. If the contents of the model are not sorted the model and
	 * view indices are the same.
	 * 
	 * @param table instance of {@link JTable}
	 * @return selected row index from the model point of view.
	 */
	public static int getModelSelectedRow(JTable table) {
		int selectedRow = table.getSelectedRow();
		return getModelRowIndex(table, selectedRow);
	}

	/**
	 * Maps the indexes of rows in terms of the view to the underlying
	 * TableModel. If the contents of the model are not sorted the model and
	 * view indices are the same.
	 * 
	 * @param table instance of {@link JTable}
	 * @return selected row indexes from the model point of view or null if no
	 *         rows are selected.
	 */
	public static int[] getModelSelectedRows(JTable table) {
		int selectedRows[] = table.getSelectedRows();
		int size = selectedRows != null ? selectedRows.length : 0;
		if (size == 0) {
			return null;
		}
		for (int i = 0; i < size; i++) {
			selectedRows[i] = getModelRowIndex(table, selectedRows[i]);
		}
		return selectedRows;
	}

	/**
	 * Maps the indexes of rows in terms of the model to view. If the contents
	 * of the model are not sorted the model and view indices are the same.
	 * 
	 * @param table instance of {@link JTable}
	 * @return selected row indexes from the view point or null if no rows are
	 *         selected.
	 */
	public static int[] getViewSelectedRows(JTable table) {
		int selectedRows[] = table.getSelectedRows();
		int size = selectedRows != null ? selectedRows.length : 0;
		if (size == 0) {
			return null;
		}
		for (int i = 0; i < size; i++) {
			selectedRows[i] = getViewRowIndex(table, selectedRows[i]);
		}
		return selectedRows;
	}

	/**
	 * Maps the index of the row in terms of the view to the underlying
	 * TableModel. If the contents of the model are not sorted the model and
	 * view indices are the same.
	 * 
	 * @param table instance of {@link JTable}
	 * @param rowIndex the index of the row in terms of the view
	 * @return selected row index from the model point of view.
	 */
	public static int getModelRowIndex(JTable table, int rowIndex) {
		if (table == null) {
			return -1;
		}
		if (rowIndex < 0) {
			return rowIndex;
		}
		try {
			return table.convertRowIndexToModel(rowIndex);
		} catch (Exception e) {
			return -1;
		}
	}

	/**
	 * Maps the index of the row in terms of the TableModel to the view. If the
	 * contents of the model are not sorted the model and view indices are the
	 * same.
	 * 
	 * @param table instance of {@link JTable}
	 * @param rowIndex the index of the row in terms of the model
	 * @return selected row index from the model point of view , or -1 if the
	 *         row isn't visible
	 * 
	 */
	public static int getViewRowIndex(JTable table, int rowIndex) {
		if (table == null) {
			return -1;
		}
		if (rowIndex < 0) {
			return rowIndex;
		}
		try {
			return table.convertRowIndexToView(rowIndex);
		} catch (Exception e) {
			return -1;
		}
	}

}
