package pl.psnc.vlab.util.gui.helper;

import java.awt.Component;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.Locale;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.TableModelEvent;
import javax.swing.text.JTextComponent;
import javax.swing.text.MaskFormatter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.gui.document.NormalizedDocument;
 
/**
 * FormHelper - abstract class defines a set of utility helper methods when
 * dealing with gui.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class FormHelper {

	/** Stores instance of field: instance */
	private static NumberFormat numberFormat;

	/** Stores an instance of logger */
	@SuppressWarnings("unused")
	private static Log log = LogFactory.getLog(FormHelper.class);

	/**
	 * Get {@link TableModelEvent} from the given argument
	 * 
	 * @param modelEvent instance of table model event
	 * @return instance of {@link TableModelEvent} or <code>NULL</code> if
	 *         given value is not instance of {@link TableModelEvent}
	 */
	public static TableModelEvent getModelEvent(Object modelEvent) {
		if (!(modelEvent instanceof TableModelEvent)) {
			log.warn("No TableModelEvent found");
			return null;
		}
		return (TableModelEvent) modelEvent;
	}

	/**
	 * Checks whether the given value is null or empty string
	 * 
	 * @param value object value
	 * @return <CODE>true</CODE> if value is null or empty, <CODE>false</CODE>
	 *         otherwise
	 */
	public static boolean isNull(Object value) {
		if (value == null) {
			return true;
		}

		String sValue = value.toString();
		if (sValue == "" || "".equals(sValue) || sValue.length() == 0) {
			return true;
		}
		return false;
	}

	/**
	 * Return {@link Boolean} representation from the given object
	 * 
	 * @param value object value
	 * @return <CODE>Boolean.TRUE</CODE> if string value is not null and
	 *         equals to true, <CODE>Boolean.FALSE</CODE> otherwise
	 */
	public static Boolean getBoolean(Object value) {
		if (value == null) {
			return null;
		}
		return Boolean.parseBoolean(value.toString());
	}

	/**
	 * Return {@link Double} representation from the given object
	 * 
	 * @param value object value
	 * @return <CODE>Double</CODE> if string value is not null and can be
	 *         parsed as double, <CODE>NULL</CODE> otherwise
	 */
	public static Double getDoubleValue(Object value) {
		try {
			return Double.parseDouble(value.toString());
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * Parse the given value using default number formatter
	 * 
	 * @param value number value to be parsed
	 * @return parsed given value using default number formatter
	 */
	public static String format(Number value) {
		if (numberFormat == null) {
			numberFormat = NumberFormat.getInstance(Locale.getDefault());
			numberFormat.setGroupingUsed(true);
		}
		try {
			return numberFormat.format(value);
		} catch (Exception e) {
			return value.toString();
		}
	}

	/**
	 * Enables/disables all child components of the given component
	 * 
	 * @param component parent component
	 */
	public static void setEnabled(JComponent component, boolean enabled) {
		Component list[] = component.getComponents();
		if (list == null || list.length == 0) {
			return;
		}
		for (Component c : list) {
			c.setEnabled(enabled);
			if (c instanceof JComponent) {
				setEnabled((JComponent) c, enabled);
			}
		}
	}

	/**
	 * Set the given text on each JTextField component. The method is recursive
	 * and works for all subcomponents
	 * 
	 * @param component parent component
	 * @param text text to be set
	 * @param tooltip tooltip to set
	 */
	public static void setText(JComponent component, String text, String tooltip) {
		Component list[] = component.getComponents();
		if (list == null || list.length == 0) {
			return;
		}
		for (Component c : list) {
			c.setFocusable(false);
			if (c instanceof JTextComponent) {
				JTextComponent textComp = ((JTextComponent) c);
				textComp.setText(text);
				textComp.setToolTipText(tooltip);
				continue;
			} else if (c instanceof JSpinner) {
				JSpinner spinner = ((JSpinner) c);
				spinner.setValue(text);
				spinner.setToolTipText(tooltip);
				continue;
			}
			if (c instanceof JComponent) {
				setText((JComponent) c, text, tooltip);
			}
		}
	}

	/**
	 * Get value of the formatted text field. If mask formatter is set it
	 * returns formatted value. Otherwise it return text value
	 * 
	 * @param component formatted text component
	 */
	public static String getValue(JTextComponent component) {
		log.debug("<getValue>");
		if (component == null) {
			return null;
		}
		Object value = null;
		if (component instanceof JFormattedTextField) {
			JFormattedTextField fTf = (JFormattedTextField) component;
			value = fTf.getFormatter() != null ? fTf.getValue() : fTf.getText();

		} else {
			value = component.getText();
		}
		return value != null ? value.toString() : null;
	}

	/**
	 * Create a {@link MaskFormatter} for the given mask and place holder
	 * 
	 * @param mask mask. Please refer to {@link MaskFormatter} for the available
	 *            choices
	 * @param placeHolder place holder character
	 * @return new instance of {@link MaskFormatter} or null if Mask Formatter
	 *         creation failed
	 */
	public static MaskFormatter getMaskFormatter(String mask, Character placeHolder) {
		if (mask == null) {
			return null;
		}
		MaskFormatter mf = null;
		try {
			mf = new MaskFormatter(mask);
			if (placeHolder != null) {
				mf.setPlaceholderCharacter(placeHolder);
			}

		} catch (ParseException e) {
			log.error("Error while creating mask. Details:" + e.getMessage());
			return null;
		}
		return mf;
	}

	/**
	 * Removes mask formatter from the given formatted text field
	 * 
	 * @param fTextField instance of {@link JFormattedTextField}
	 */
	public static void removeMaskFormatter(JFormattedTextField fTextField) {
		if (fTextField == null) {
			return;
		}
		fTextField.setFormatterFactory(null);
		fTextField.setValue("");
		fTextField.setToolTipText(null);
	}

	/**
	 * Removes mask formatter from the given spinner
	 * 
	 * @param spinner instance of {@link JSpinner}
	 */
	public static void removeMaskFormatter(JSpinner spinner) {
		if (spinner == null) {
			return;
		}
		JComponent editor = spinner.getEditor();
		if (editor instanceof JSpinner.DefaultEditor) {
			JSpinner.DefaultEditor spEditor = (JSpinner.DefaultEditor) editor;
			JFormattedTextField textField = spEditor.getTextField();
			removeMaskFormatter(textField);
		}
		spinner.setValue(null);
	}

	/**
	 * Set {@link NormalizedDocument} on the given text component
	 * 
	 * @param component text component
	 */
	public static void setNormalizedDocument(JTextComponent component, int maxLenght) {
		if (component == null) {
			return;
		}
		NormalizedDocument doc = new NormalizedDocument(component, maxLenght);
		component.setDocument(doc);
	}

	/**
	 * Selects the given row index at the given table.
	 * 
	 * @param row row index
	 * @param table instance of {@link JTable}
	 */
	public static void setSelectedRow(Integer row, JTable table) {
		if (table == null) {
			log.debug("<setSelectedRow: table is NULL");
			return;
		}
		if (row == null) {
			log.debug("Row is NULL, set to 0");
			row = 0;
		}
		row = TableHelper.getViewRowIndex(table, row);
		log.debug(String.format("<setSelectedRow: [row=%s,table=%s]", row, table.getName()));
		int rowCount = table.getModel().getRowCount();
		if (row < 0 || row >= rowCount) {
			log.debug("Row index adjsuted to 0");
			row = 0;
		}
		table.changeSelection(row, 0, false, false);
		log.debug("</setSelectedRow: OK>");
	}

	/**
	 * Removes rows and columns selection from the given table.
	 * 
	 * @param table instance of {@link JTable}
	 */
	public static void removeSelection(JTable table) {
		log.debug("<removeSelection>");
		if (table == null) {
			log.debug("</removeSelection: table is NULL");
			return;
		}
		table.clearSelection();
		log.debug("</removeSelection>");
	}

	/**
	 * Set max value for the given spinners
	 * 
	 * @param max max value
	 * @param spinners spinner list
	 */
	public static void setMax(double max, JSpinner... spinners) {
		if (spinners == null) {
			return;
		}
		for (JSpinner spinner : spinners) {
			SpinnerModel model = spinner.getModel();
			if (model instanceof SpinnerNumberModel) {
				SpinnerNumberModel nModel = (SpinnerNumberModel) model;
				nModel.setMaximum(max);
			}
		} // end of for
	}

	/**
	 * Set current value for the given spinners
	 * 
	 * @param value max value
	 * @param spinners spinner list
	 */
	public static void setValue(double value, JSpinner... spinners) {
		if (spinners == null) {
			return;
		}
		for (JSpinner spinner : spinners) {
			SpinnerModel model = spinner.getModel();
			model.setValue(value);
		} // end of for
	}
}