/*
 * NumberDocument.java
 *
 * 
 */

package pl.psnc.vlab.util.gui.document;

import java.awt.Toolkit;
import java.awt.event.KeyEvent;

import javax.swing.JComponent;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.DocumentEvent.EventType;
import javax.swing.text.AbstractDocument;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.gui.helper.FormHelper;
import pl.psnc.vlab.util.i18n.ResourceBundleManager;

/**
 * {@link NormalizedDocument} class - document is a container for text that
 * serves as the model for swing text components. The document accepts only
 * normalised characters and numbers.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class NormalizedDocument extends PlainDocument {

	/** Bundle name */
	private static final String BUNDLE = "bundle/document";

	/** Default Serial uid */
	private static final long serialVersionUID = 1L;

	/** Logger instance */
	private Log log = LogFactory.getLog(this.getClass().getName());

	/** Maximum length of the characters that can be inserted */
	private int maxLength;

	/** Stores tooltip text: max exceeded */
	private String maxLengthExceeded;

	/** Stores tooltip text: invalid character */
	private static String invalidCharacter;

	/**
	 * Creates a new instance of NumberDocument
	 * 
	 * @param component the source component
	 * @param maxLength maximum number of characters that may be inserted
	 */
	public NormalizedDocument(JComponent component, int maxLength) {
		this.maxLength = maxLength;
		try {
			ResourceBundleManager rbm = new ResourceBundleManager(BUNDLE);
			invalidCharacter = rbm.getValue("plain.document.invalid.character");
			maxLengthExceeded = rbm.getFormattedValue("plain.document.length.exceeded",
					new Object[] { String.valueOf(maxLength) });
		} catch (Exception e) {
			log.error(e.getMessage());
		}

		addDocumentListener(new InputListener(component));
	}

	/**
	 * This inner class represents the DocumentEvent, which is fired whenever
	 * not allowed characters are inserted.
	 */
	private class UpdateTooltipDocEvent extends AbstractDocument.DefaultDocumentEvent {

		/** Default serial uid */
		private static final long serialVersionUID = 1L;

		public UpdateTooltipDocEvent(int offset, int lenght, EventType type, String toolTip) {
			super(offset, lenght, type);
			this.toolTip = toolTip;
		}

		/**
		 * Holds value of property toolTip.
		 */
		private String toolTip;

		/**
		 * Getter for property toolTip.
		 * 
		 * @return Value of property toolTip.
		 */
		public String getToolTip() {
			return this.toolTip;
		}

	}

	/**
	 * Inserts a string of content.
	 * 
	 * @param offset the offset into the document to insert the content >= 0.
	 *        All positions that track change at or after the given location
	 *        will move.
	 * @param str the string to insert
	 * @param attr the attributes to associate with the inserted content. This may
	 *        be null if there are no attributes.
	 * @exception BadLocationException the given insert position is not a valid
	 *            position within the document
	 */
	public void insertString(int offset, String str, javax.swing.text.AttributeSet attr)
			throws javax.swing.text.BadLocationException {
		if (getLength() + str.length() > maxLength) {
			Toolkit.getDefaultToolkit().beep();
			fireChangedUpdate(new UpdateTooltipDocEvent(offset, getLength(), EventType.CHANGE,
					maxLengthExceeded));
			return;
		}

		int size = str.length();
		StringBuffer result = new StringBuffer(size);

		for (int i = 0; i < size; i++) {
			char ch = str.charAt(i);
			if (Character.isWhitespace(ch) && offset == 0) {
				fireChangedUpdate(new UpdateTooltipDocEvent(offset, getLength(), EventType.CHANGE,
						invalidCharacter));
				continue;
			}

			if (Character.isLetterOrDigit(ch) || ch == '_' || ch == '-'
					|| Character.isWhitespace(ch)) {
				result.append(ch);
			} else {
				fireChangedUpdate(new UpdateTooltipDocEvent(offset, getLength(), EventType.CHANGE,
						invalidCharacter));
			}
		} // end of for

		String value = result.toString();
		if (!FormHelper.isNull(value)) {
			super.insertString(offset, str, attr);
		}
		return;
	}

	// ------------------------------------------------------------------------
	// ---- Private Helper classes

	/**
	 * {@link InputListener} class - responsible for listening on input key
	 * strokes
	 * 
	 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
	 * @email osa@man.poznan.pl
	 * 
	 */
	private class InputListener implements DocumentListener {

		/** Stores instance of source component */
		JComponent component;

		/**
		 * Crates a new instance of Input listener
		 * 
		 * @param component
		 */
		InputListener(final JComponent component) {
			this.component = component;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.event.DocumentListener#changedUpdate(javax.swing.event.DocumentEvent)
		 */
		@Override
		public void changedUpdate(DocumentEvent e) {
			NormalizedDocument.UpdateTooltipDocEvent event = (NormalizedDocument.UpdateTooltipDocEvent) e;
			component.setToolTipText(event.getToolTip());
			KeyEvent kEvent = new KeyEvent(component, KeyEvent.KEY_PRESSED, 0, KeyEvent.CTRL_MASK,
					KeyEvent.VK_F1, KeyEvent.CHAR_UNDEFINED);
			component.dispatchEvent(kEvent);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.event.DocumentListener#insertUpdate(javax.swing.event.DocumentEvent)
		 */
		@Override
		public void insertUpdate(DocumentEvent arg0) {
			// nothing to do here
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.event.DocumentListener#removeUpdate(javax.swing.event.DocumentEvent)
		 */
		@Override
		public void removeUpdate(DocumentEvent arg0) {
			// nothing to do here
		}
	}

}
