/*
 * ActionHelper.java
 *
 * Created on December 16, 2005, 1:04 PM
 *
 */

package pl.psnc.vlab.util.gui.action.helper;

import java.util.Observable;
import java.util.Observer;

import javax.swing.AbstractButton;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JToggleButton;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.gui.action.AbstractActionDefault;
import pl.psnc.vlab.util.gui.action.factory.ActionFactory;
import pl.psnc.vlab.util.gui.action.factory.ActionType;

/**
 * Class contains a set of utility methods for the Swing components
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public final class ActionHelper {

	/** Instance of logger */
	private static Log log = LogFactory.getLog(ActionHelper.class.getName());

	/**
	 * Set action for the given component. The tooltip, icons and text are
	 * copied from the given action. If the given component is of type JButton
	 * the empty text will be set
	 * 
	 * @param component swing component, instance of {@link AbstractButton}
	 * @param action instance of action {@link AbstractActionDefault}
	 * @param enabled specifies whether action is enabled
	 */
	public static void setAction(AbstractButton component, AbstractActionDefault action,
			boolean enabled) {
		component.setAction(action);
		action.setEnabled(enabled);
		component.setToolTipText(action.getShortDescription());
		removeName(component);
		// set disabled icon
		ImageIcon disabledActionIcon = action.getDisabledActionIcon();
		if (disabledActionIcon == null) {
			log.debug("Disabled icon not defined");
			return;
		}
		component.setDisabledIcon(disabledActionIcon);
	}

	/**
	 * Set action for the given component. The tooltip, icons and text are
	 * copied from the given action. If the given component is of type JButton
	 * the empty text will be set
	 * 
	 * @param component swing component, instance of {@link AbstractButton}
	 * @param actionType action type, instance {@link ActionType}
	 * @param enabled specifies whether action is enabled
	 */
	public static void setAction(AbstractButton component, ActionType actionType, boolean enabled) {
		if (actionType == null) {
			return;
		}
		AbstractActionDefault action = ActionFactory.getAction(actionType);
		setAction(component, action, enabled);
	}

	/**
	 * Add action observer to the given observable object
	 * 
	 * @param observable observable instance {@link Observable}
	 * @param observer observer instance {@link Observer}
	 */
	public static void addObserver(ActionType observable, ActionType observer) {
		log.debug("<addObserver>");
		if (observable == null || observer == null) {
			log.debug("</addObserver: NULL>");
			return;
		}

		Observable source = ActionFactory.getObservable(observable);
		Observer destination = ActionFactory.getObserver(observer);

		if (source == null || destination == null) {
			log.debug("</addObserver: NULL>");
			return;
		}
		source.addObserver(destination);
		log.debug("</addObserver>");
	}

	// ----------------------------------------------------------------------------------------
	// ---- Private Helper Methods

	/**
	 * Remove name from the given component
	 * 
	 * @param component
	 */
	private static void removeName(AbstractButton component) {
		if (component instanceof JButton || component instanceof JToggleButton) {
			component.setText("");
		}
	}

}
