package pl.psnc.vlab.util.gui.action;

import java.util.Observable;
import java.util.Observer;

/**
 * {@link LocalizedObservableAbstractAction} class - provides default
 * implementation for the JFC Action interface with the localised text
 * descriptions and possibility of adding observers.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class LocalizedObservableAbstractAction extends LocalizedAbstractAction {

	/** Inner class - stores Observable instance */
	private ModelObservable observable;

	/**
	 * Define an Action Object with a specified action name, icon and short
	 * description.
	 * 
	 * @param nameKey name of the action
	 * @param iconPath the icon for the current action
	 * @param description action short description
	 * 
	 */
	public LocalizedObservableAbstractAction(String nameKey, String iconPath, String description) {
		super(nameKey, iconPath, description);
		this.observable = new ModelObservable();
	}

	/**
	 * Adds an observer to the set of observers for this object, provided that
	 * it is not the same as some observer already in the set. The order in
	 * which notifications will be delivered to multiple observers is not
	 * specified. See the class comment.
	 * 
	 * @param observer an observer to be added.
	 */
	public void addObserver(Observer observer) {
		if (observer != null) {
			this.observable.addObserver(observer);
		}
	}

	/**
	 * Deletes an observer from the set of observers of this object. Passing
	 * <CODE>null</CODE> to this method will have no effect.
	 * 
	 * @param observer the observer to be deleted.
	 */
	public void deleteObserver(Observer observer) {
		this.observable.deleteObserver(observer);
	}

	/**
	 * If this object has changed, as indicated by the <code>hasChanged</code>
	 * method, then notify all of its observers and then call the
	 * <code>clearChanged</code> method to indicate that this object has no
	 * longer changed.
	 * <p>
	 * Each observer has its <code>update</code> method called with two
	 * arguments: this observable object and the <code>arg</code> argument.
	 * 
	 * @param arg any object.
	 * @see java.util.Observable#clearChanged()
	 * @see java.util.Observable#hasChanged()
	 * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
	 */
	public void notifyObservers(Object arg) {
		this.observable.setChanged();
		this.observable.notifyObservers(arg);
	}

	// ----------------------------------------------------------------------------------
	// ---- Private Helper Methods

	/**
	 * {@link ModelObservable} class - observable model
	 * 
	 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
	 * @email osa@man.poznan.pl
	 * 
	 */
	public class ModelObservable extends Observable {

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.util.Observable#setChanged()
		 */
		@Override
		public synchronized void setChanged() {
			super.setChanged();
		}
	}

	// ----------------------------------------------------------------------------------
	// ---- Setters / Getters

	/**
	 * Get value of field: observable
	 * 
	 * @return the observable
	 */
	public ModelObservable getObservable() {
		return observable;
	}
}
