package pl.psnc.graph.model.marquee;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jgraph.JGraph;
import org.jgraph.graph.BasicMarqueeHandler;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.PortView;

import pl.psnc.graph.gui.popup.JCustomGraphPopup;
import pl.psnc.graph.util.GraphHelper;
import pl.psnc.vlab.exception.ResourceNotFoundException;
import pl.psnc.vlab.util.gui.message.JMessage;

/**
 * {@link CustomMarqueeHandler} class
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class CustomMarqueeHandler extends BasicMarqueeHandler {

	/** Stores instance of field: log */
	Log log = LogFactory.getLog(CustomMarqueeHandler.class);

	/** Holds the Start and the Current Point */
	protected Point2D startPoint, currentPort;

	/** Holds the First and the Current Port */
	protected PortView currentPortView, firstPortView;

	/** Stores instance of field: graph */
	protected final JGraph graph;

	/**
	 * Creates a new instance of {@link CustomMarqueeHandler}
	 * 
	 * @param graph graph instance
	 */
	public CustomMarqueeHandler(JGraph graph) {
		this.graph = graph;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jgraph.graph.BasicMarqueeHandler#isForceMarqueeEvent(java.awt.event.MouseEvent)
	 */
	public boolean isForceMarqueeEvent(MouseEvent e) {
		if (e.isShiftDown()) {
			return false;
		}

		if (SwingUtilities.isRightMouseButton(e)) {
			return true;
		}

		currentPortView = GraphHelper.getSourcePortAt(e.getPoint(), graph);

		if (currentPortView != null && graph.isPortsVisible()) {
			// port found and in connect mode
			return true;
		}
		return super.isForceMarqueeEvent(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jgraph.graph.BasicMarqueeHandler#mousePressed(java.awt.event.MouseEvent)
	 */
	public void mousePressed(final MouseEvent e) {
		if (SwingUtilities.isRightMouseButton(e)) {
			try {
				JPopupMenu menu = new JCustomGraphPopup();
				menu.show(graph, e.getX(), e.getY());
			} catch (ResourceNotFoundException error) {
				JMessage.showErrorMessage(graph, error);
			}
		} else if (currentPortView != null && graph.isPortsVisible()) {
			// Remember Start Location
			startPoint = graph.toScreen(currentPortView.getLocation());
			// Remember First Port
			firstPortView = currentPortView;
		} else {
			// Call Superclass
			super.mousePressed(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jgraph.graph.BasicMarqueeHandler#mouseDragged(java.awt.event.MouseEvent)
	 */
	public void mouseDragged(MouseEvent e) {
		if (startPoint != null) {
			Graphics g = graph.getGraphics();
			PortView newPort = GraphHelper.getTargetPortAt(e.getPoint(), graph);

			if (newPort == null || newPort != currentPortView) {
				paintConnector(Color.black, graph.getBackground(), g);
				currentPortView = newPort;
				if (currentPortView != null) {
					currentPoint = graph.toScreen(currentPortView.getLocation());
				} else {
					currentPoint = graph.snap(e.getPoint());
				}
				paintConnector(graph.getBackground(), Color.black, g);
			}
		}
		super.mouseDragged(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jgraph.graph.BasicMarqueeHandler#mouseReleased(java.awt.event.MouseEvent)
	 */
	public void mouseReleased(MouseEvent e) {
		if (e != null && currentPortView != null && firstPortView != null
				&& firstPortView != currentPortView) {
			GraphHelper.connect(graph, firstPortView, currentPortView);
			e.consume();
		} else {
			graph.repaint();
		}

		firstPortView = currentPortView = null;
		startPoint = currentPort = null;
		super.mouseReleased(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jgraph.graph.BasicMarqueeHandler#mouseMoved(java.awt.event.MouseEvent)
	 */
	public void mouseMoved(MouseEvent e) {
		// Check Mode and Find Port
		PortView sourcePortAt = GraphHelper.getSourcePortAt(e.getPoint(), graph);
		if (e != null && sourcePortAt != null && graph.isPortsVisible()) {
			graph.setCursor(new Cursor(Cursor.HAND_CURSOR));
			e.consume();
			return;
		}
		super.mouseMoved(e);
	}

	/**
	 * Paint connector
	 * 
	 * @param fg foreground colour
	 * @param bg background colour
	 * @param g instance of {@link Graphics}
	 */
	protected void paintConnector(Color fg, Color bg, Graphics g) {
		g.setColor(fg);
		g.setXORMode(bg);
		paintPort(graph.getGraphics());
		if (firstPortView != null && startPoint != null && currentPoint != null) {
			g.drawLine((int) startPoint.getX(), (int) startPoint.getY(), (int) currentPoint.getX(),
					(int) currentPoint.getY());
		}
	}

	/**
	 * Paint port
	 * @param g instance of {@link Graphics}
	 */
	protected void paintPort(Graphics g) {
		// If Current Port is Valid
		if (currentPortView != null) {
			// If Not Floating Port...
			boolean o = (GraphConstants.getOffset(currentPortView.getAllAttributes()) != null);
			// ...Then use Parent's Bounds
			Rectangle2D r = (o) ? currentPortView.getBounds() : currentPortView.getParentView()
					.getBounds();
			// Scale from Model to Screen
			r = graph.toScreen((Rectangle2D) r.clone());
			// Add Space For the Highlight Border
			r.setFrame(r.getX() - 3, r.getY() - 3, r.getWidth() + 6, r.getHeight() + 6);
			// Paint Port in Preview (=Highlight) Mode
			graph.getUI().paintCell(g, currentPortView, r, true);
		}
	}

}
