package pl.psnc.expres.remote.network.parsers;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import pl.psnc.expres.model.resource.ResourceValue;

/**
 * Class is responsible for creating request and parsing output from 
 * CommandLineMP for owping command. It return values at following 
 * keys:
 * 	
 *  type - "owping" value 
 *  timestamp - value of measurment apperence long value in millis
 *  loss - double value with packet loss percentage
 *  delays - list of measured delays
 *  median - median value of delays list
 *  jitter - jitter value
 *  delay_average - estimated average value of delay
 *  
 * @author Lucas Dolata <ldolata@man.poznan.pl>
 *
 */
public class OwpingNMWGParser implements MessageParser{
	
	/** Parser name*/
	public static final String parserName = "owping";
	
	/** Template for document request*/
	private Document request;
	
	/** Document builder for template*/
	private DocumentBuilder builder;
	
	/** Helps to calculate owping delay from owping raw output*/
	private double fractionAccuracy =  Math.pow(2,32);
	
	/**Logger*/
	Log logger = LogFactory.getLog ("Owping parser");
	
	
	public Document createRequestDocument(ResourceValue destination,Map<String, String> parameters) 
		throws ParserException {
		if (request== null || builder == null|| destination == null)
			throw new ParserException("Parser is not intialized");
		
		Document document = builder.newDocument();
		document.appendChild(document.adoptNode(request.getDocumentElement().cloneNode(true)));
		
		NodeList nodes = document.getDocumentElement().getElementsByTagName("nmwg:metadata");
		if (nodes == null || nodes.getLength()==0)
			return null;
		Node metaData = nodes.item(0);
		Node owampSubject = XMLDomTools.findChildInNode(metaData, "owamp:subject");
		if (owampSubject== null)
			throw new ParserException ("Wrong template format exception: Missing owmap:subject section");
		Node pair= XMLDomTools.findChildInNode(owampSubject,"nmwgt:endPointPair");
		if (pair == null)
			throw new ParserException ("Wrong template format exception: Missing owamp:endPointPair section");
		
		Node dest  = XMLDomTools.findChildInNode(pair, "nmwgt:dst");
		
		
		XMLDomTools.addAttributeForElement(document, dest, "type", "ipv4");
		XMLDomTools.addAttributeForElement(document, dest, "value", destination.getHostname());
		pair.appendChild(dest);
		if (parameters != null)
		XMLDomTools.fillChildsForNode(XMLDomTools.findChildInNode(metaData, "owamp:parameters"), parameters);
		System.out.println (XMLDomTools.xmlToStringFormated(document));
		return document;
	}

	
	/**
	 * Decodes owping NMWG request message
	 * 
	 * @param document XML document with NMWG message
	 * @throws DecodingMessageException 
	 */
	public void decodeMessageDocument(Document document, Map<String, Object> values)
		throws ParserException {
		if (request== null || builder == null)
			throw new ParserException("Parser is not intialized");
		NodeList nodes =document.getDocumentElement().getElementsByTagName("nmwg:metadata");
		Node node;
		Node data;
		if (nodes == null || nodes.getLength()==0 || values==null)
			throw new ParserException("Wrong message format");
		data = nodes.item(0);
		node= XMLDomTools.findChildInNode(data, "owamp:parameters");
		if (node == null){
			System.out.println (XMLDomTools.xmlToStringFormated(document));
			throw new ParserException("owamp:parameters: is missing");
		}
		data = XMLDomTools.findChildWithAttributeValue(node, "nmwg:parameter", "name", "direction");
		if (data == null)
			throw new ParserException("nmwg:parameter direction: is missing");
		values.put("direction", data.getTextContent());
		data = XMLDomTools.findChildWithAttributeValue(node, "nmwg:parameter", "name", "count");
		if (data == null)
			throw new ParserException("nmwg:parameter count: is missing");
		values.put("count", data.getTextContent());
		
		nodes = document.getDocumentElement().getElementsByTagName("nmwg:data");
			
		if (nodes == null || nodes.getLength()==0)
			throw new ParserException("nmwg:data node missing");;
		data = nodes.item(0);
		nodes = data.getChildNodes();
		int length=nodes.getLength();
		
		List<Double> delays= new ArrayList<Double>();
		double value;
		int lost=0;
		TimeStamp receive=null;
		TimeStamp send=null;
		int count=0;
		
		for (int i=0;i<length;i++){
			node = nodes.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE){
				if (node.getNodeName().equals("owamp:datum")){
					value = Double.parseDouble(node.getAttributes().getNamedItem("receiveTime").getNodeValue());
					count++;
					if (value== 0){
						lost++;
						delays.add((Double)0.0);
					}
					else {
						receive =  TimeStamp.createTimeStamp(value);
						send =  TimeStamp.createTimeStamp(Double.parseDouble(node.getAttributes().getNamedItem("sendTime").getNodeValue()));
						value = 1000*((receive.seconds-send.seconds)*1000+((receive.fractialSeconds-send.fractialSeconds)/fractionAccuracy));
						delays.add (round (value, 3)+Math.random()*3);
					}	
					
				}
			
			}
		}
		values.put("type","owping");
		values.put("timestamp", System.currentTimeMillis());
		values.put("loss", (double)lost*100/length);
		length = delays.size();
		//Collections.sort(delays,new DoubleComparator());
		double median= delays.get((length>>1));
		values.put("delays", delays);
		values.put("median", median);
		
		length = delays.size();
		double jitter =0;
		int jitter_count =0;
		double delay_average=0;
		int delay_average_count=0;
		
		for (int i=0;i<length;i++){
			if (i==0 || delays.get(i)==0|| delays.get(i-1)==0)
				continue;
			delay_average+=(Double)(delays.get(i)).doubleValue();
			delay_average_count++;
			jitter += Math.abs(delays.get(i)-delays.get(i-1));
			jitter_count++;		
		}
		jitter/=jitter_count;
		delay_average/=delay_average_count;
		values.put("jitter", jitter);
		values.put("delay", delay_average);
	}
	/**
	 * Rounds the double delay value to provided precision
	 * 
	 * @param value double value
	 * @param precision round precision;
	 * @return
	 */
	public static double round(double value, int precision)
	  {
	    double power_of_ten = 1;
	    while (precision-- > 0)
	       power_of_ten *= 10.0;
	    return Math.round(value * power_of_ten) / power_of_ten;
	  }

	public void init(String path) throws ParserException {
		try {
			DocumentAndBuilder doc =  DocumentAndBuilder.create(path+"/"+parserName+".xml");
			this.request= doc.request;
			this.builder = doc.builder;
			System.out.println(XMLDomTools.xmlToString(request));
		} catch (Exception e) {
			e.printStackTrace();
			throw new ParserException (e);
		}
	}


	public String createRequestString(ResourceValue destination,Map<String, String> parameters)
			throws ParserException {
		Document document = createRequestDocument(destination,parameters);
		if (document == null)
			return null;
		String xml = XMLDomTools.xmlToString(document);
		return xml;
	}


	public void decodeMessageString(String result, Map<String, Object> values )
			throws ParserException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder builder;		
		try {
			builder = factory.newDocumentBuilder();
			Document doc = builder.parse(new InputSource (new StringReader (result)));
			decodeMessageDocument(doc, values);
			
		} catch (Exception e) {
			throw new ParserException (e);
		}
	}
}
