package pl.psnc.expres.remote.network.parsers;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import pl.psnc.expres.model.resource.ResourceValue;

/**
 * Class is responsible for creating requests an parsing result of BWCL NMWG Message
 * 
 * @author Lucas Dolata <ldolata@man.poznan.pl>
 *
 */
public class BwctlNMWGParser implements MessageParser {
	/**
	 * Identifies the name of the parser
	 */
	public static final String parserName = "bwctl";
	/**
	 * Identifies a template of an request
	 */
	private Document request;
	/**
	 * Identifies a builder for document modification 
	 */
	private DocumentBuilder builder;
	
	
	public Document createRequestDocument(ResourceValue destination , Map<String, String> parameters)
			throws ParserException {
		if (request== null || destination == null)
			return null;  
		Document document = builder.newDocument();
	    document.appendChild(document.adoptNode(request.getDocumentElement().cloneNode(true)));
		NodeList nodes = document.getDocumentElement().getElementsByTagName("nmwg:metadata");
		if (nodes == null || nodes.getLength()==0)
			return null;
		Node metaData = nodes.item(0);
		Node owampSubject = XMLDomTools.findChildInNode(metaData, "bwctl:subject");
		if (owampSubject== null)
			throw new ParserException ("Wrong template format exception: Missing bwctl:subject section");
		Node pair= XMLDomTools.findChildInNode(owampSubject,"nmwgt:endPointPair");
		if (pair == null)
			throw new ParserException ("Wrong template format exception: Missing owamp:endPointPair section");
		Node dest  = XMLDomTools.findChildInNode(pair, "nmwgt:dst");
		XMLDomTools.addAttributeForElement(document, dest, "type", "ipv4");
		XMLDomTools.addAttributeForElement(document, dest, "value", destination.getHostname());
		pair.appendChild(dest);
		XMLDomTools.fillChildsForNode(XMLDomTools.findChildInNode(metaData, "bwctl:parameters"), parameters);
		return document;
	}

	public void decodeMessageDocument(Document document, Map<String,Object> values) throws ParserException {
		if (document == null)
			throw new ParserException("Parsing document is null"); 
		NodeList nodes = document.getDocumentElement().getElementsByTagName("nmwg:data");
		NodeList data;
		if (nodes == null || values==null)
			throw new ParserException("No nmwg:data node");
		
		List<Long> receivers= new ArrayList<Long>();
		List<Long> senders= new ArrayList<Long>();
		int length = nodes.getLength();
		List<Long>current;
		for (int i=0;i<length;i++){
			current =null;
			if (nodes.item(i).getAttributes().getNamedItem("id").getNodeValue().indexOf("sender")!=-1)
				current = senders;
			if (nodes.item(i).getAttributes().getNamedItem("id").getNodeValue().indexOf("receiver")!=-1)
				current = receivers;
			
			if (current== null) continue;
			data = nodes.item(i).getChildNodes();
			Node node=null;
			int dataLength = data.getLength();
			long value;
			String valueUnit;
			
			for (int k=0;k<dataLength;k++){
					node = data.item(k);
					if (node.getNodeType()==Node.ELEMENT_NODE){
						if (node.getNodeName().equals("bwctl:datum")){
							value = Long.parseLong(node.getAttributes().getNamedItem("value").getNodeValue()); 
							valueUnit=node.getAttributes().getNamedItem("valueUnits").getNodeValue();
							if (valueUnit.equals("bytes/sec"))
									value *= 8;
							current.add(value);
						}
					}
			}
		}
		if (receivers.isEmpty() && senders.isEmpty()){
			System.out.print(XMLDomTools.xmlToStringFormated(document));
			throw new ParserException ("Empty measured values");
		}
		
		if (!receivers.isEmpty()){
			values.put("receiver-bandwidths", receivers);
			long receiver=0;
			int  count=0;
			for (Long value:receivers)
				if (value.longValue()!= 0){
					receiver+=value.longValue();
					count++;
				}
			receiver/=count;
			values.put("receiver-bandwidth", receiver);
		}
		if (!senders.isEmpty()){
			values.put("sender-bandwidths", senders);
			long sender=0;
			int count=0;
			for (Long value:senders)
				if (value.longValue()!= 0){
					sender+=value.longValue();
					count++;
				}
			if (count >0)
			sender /=count;
			values.put("sender-bandwidth", sender);
		}
	}

	public void init(String path) throws ParserException {
		try {
			DocumentAndBuilder doc =  DocumentAndBuilder.create(path+"/"+parserName+".xml");
			this.request= doc.request;
			this.builder = doc.builder;
		} catch (Exception e) {
			throw new ParserException (e);
		}
	}

	public String createRequestString(ResourceValue destination,Map<String, String> parameters)
			throws ParserException {
		Document doc =createRequestDocument(destination,parameters);
		return XMLDomTools.xmlToString(doc);
	}

	public void decodeMessageString(String result, Map<String, Object> resultMap)throws ParserException{
			
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;		
		try {
			builder = factory.newDocumentBuilder();
			Document doc = builder.parse(new InputSource (new StringReader (result)));
			decodeMessageDocument(doc, resultMap);
		} catch (Exception e) {
			throw new ParserException (e);
		}
	}

}
