/**
 * ResourceReader.java
 * Created on Nov 12, 2007, 2:53:59 PM
 */
package pl.psnc.expres.model.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

import pl.psnc.vlab.exception.ResourceNotFoundException;

import com.sdicons.json.mapper.JSONMapper;
import com.sdicons.json.mapper.MapperException;
import com.sdicons.json.model.JSONValue;
import com.sdicons.json.parser.JSONParser;

/**
 * {@link JsonReaderHelper} class - used to load and parse json format resources
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class JsonReaderHelper {

	/**
	 * Load json resource and map it to the given java object (resource mapper)
	 * 
	 * @param fileName File object representing a Json file with resource
	 *        description
	 * @param resourceMapper java class which will be used to map json to java
	 *        object
	 * @return java representation of json resource
	 * @throws ResourceNotFoundException
	 */
	public static Object load(File fileName, Class<?> resourceMapper)
			throws ResourceNotFoundException {
		try {
			InputStream inStream = new FileInputStream(fileName);
			return load(inStream, resourceMapper);
		} catch (FileNotFoundException err) {
			throw new ResourceNotFoundException(err.getMessage(), err);
		}
	}

	/**
	 * Load json resource and map it to the given java object (resource mapper)
	 * 
	 * @param fileName file containing json resource
	 * @param resourceMapper java class which will be used to map json to java
	 *        object
	 * @return java representation of json resource
	 * @throws ResourceNotFoundException
	 */
	public static Object load(String fileName, Class<?> resourceMapper)
			throws ResourceNotFoundException {
		return load(new File(fileName), resourceMapper);
	}

	/**
	 * Load json resource and map it to the given java object (resource mapper)
	 * 
	 * @param input input stream with json resource
	 * @param resourceMapper java class which will be used to map json to java
	 *        object
	 * @return java representation of json resource
	 * @throws ResourceNotFoundException
	 */
	public static Object load(InputStream input, Class<?> resourceMapper)
			throws ResourceNotFoundException {
		try {
			final JSONParser jsonParser = new JSONParser(input);
			JSONValue lMyObject = jsonParser.nextValue();
			return JSONMapper.toJava(lMyObject, resourceMapper);
		} catch (Exception err) {
			// TODO add internationalization
			throw new ResourceNotFoundException("Error while loading resource for mapper: "
					+ resourceMapper + ". Details: " + err.getMessage(), err);
		}
	}

	/**
	 * Convert pojo object to JSON representation
	 * 
	 * @param javaObject java object
	 * @return JSON representation of java pojo object
	 * @throws MapperException thrown when conversion fails
	 */
	public static String jsonToString(Object javaObject) throws MapperException {
		JSONValue jsonValue = JSONMapper.toJSON(javaObject);
		return jsonValue.render(false);
	}
}
