/**
 * VlbiExperimentFactory.java
 * Created on Nov 8, 2007, 3:42:13 PM
 */
package pl.psnc.expres.model.factory;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

import javax.activation.DataHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.expres.model.experiment.VlbiExperiment;
import pl.psnc.expres.model.factory.helper.VlabiExperimentHelper;
import pl.psnc.expres.model.util.JsonReaderHelper;
import pl.psnc.expres.remote.broker.xsd.RemoteVlbiExperiment;
import pl.psnc.vlab.exception.ResourceNotFoundException;
import pl.psnc.vlab.util.FormTools;

/**
 * {@link VlbiExperimentFactory} class - the factory is used to load a VLBI
 * experiment resources (vex, ccf, workflow, jgx).
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class VlbiExperimentFactory {

	/** Logger instance */
	private static Log log = LogFactory.getLog(VlbiExperimentFactory.class);

	/** Stores an instance of VlbiExperiment */
	private static VlbiExperiment instance;

	/**
	 * Get instance of {@link VlbiExperiment}
	 * 
	 * @return instance of {@link VlbiExperiment}
	 */
	public static VlbiExperiment getInstance() {
		if (instance == null) {
			instance = new VlbiExperiment();
		}
		return instance;
	}

	/**
	 * Creates a new instance of {@link VlbiExperiment}
	 * 
	 * @return new instance of {@link VlbiExperiment}
	 */
	public static VlbiExperiment newInstance() {
		instance = new VlbiExperiment();
		return instance;
	}

	/**
	 * Creates new instance of vlbi experiment based on remote vlbi experiment
	 * (instance of {@link RemoteVlbiExperiment})
	 * 
	 * @param remoteVlbiExp remote vlbi experiment (instance of
	 *        {@link RemoteVlbiExperiment})
	 * @return new instance of vlbi experiment ( {@link VlbiExperiment} )
	 */
	public static VlbiExperiment getInstance(RemoteVlbiExperiment remoteVlbiExp)
			throws ResourceNotFoundException {
		log.debug("<getInstance>");
		if (remoteVlbiExp == null) {
			log.debug("</getInstance: NULL>");
			return null;
		}

		DataHandler dataHandler = remoteVlbiExp.getVlbiExperiment();
		InputStream inputStream;
		try {
			inputStream = dataHandler.getInputStream();

			// encode vlbi experiment
			VlbiExperiment vlbiExperiment = VlbiExperimentFactory.getInstance(inputStream);			
			VlbiExperiment converted = VlabiExperimentHelper.convert(vlbiExperiment, remoteVlbiExp);
			inputStream.close();
			return converted;
		} catch (Exception e) {
			log.debug("</getInstance: ERROR>");
			throw new ResourceNotFoundException(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * Creates a VlbiExperiment model from the given file
	 * 
	 * @param file content of vlbi experiment
	 * @return instance of {@link VlbiExperiment}
	 * @throws ResourceNotFoundException thrown when model creation fails
	 */
	public static VlbiExperiment getInstance(String file) throws ResourceNotFoundException {
		log.debug("<getVlbiExperiment>");
		if (FormTools.isValidateFormStringNull(file)) {
			return null;
		}
		FileInputStream inputStream;
		try {
			inputStream = new FileInputStream(new File(file));
		} catch (Exception e) {
			throw new ResourceNotFoundException(e.getLocalizedMessage(), e);
		}
		return getInstance(inputStream);
	}

	/**
	 * Creates a VlbiExperiment model from the given input stream.
	 * 
	 * @param inStream content of vlbi experiment
	 * @return instance of {@link VlbiExperiment}
	 * @throws ResourceNotFoundException thrown when model creation fails
	 */
	public static VlbiExperiment getInstance(InputStream inStream) throws ResourceNotFoundException {
		log.debug("<getVlbiExperiment>");

		Object res = JsonReaderHelper.load(inStream, VlbiExperiment.class);
		if (res == null) {
			// TODO add i18n
			throw new ResourceNotFoundException("Error creating VlbiExperiment model");
		}
		instance = ((VlbiExperiment) res);
		log.debug("</getVlbiExperiment>");
		return instance;
	}
}
