package pl.psnc.vlab.util.thread;

import java.awt.Component;
import java.util.List;

import javax.swing.Action;
import javax.swing.SwingWorker;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.gui.message.JMessage;
import pl.psnc.vlab.util.gui.progress.ProgressTask;
import pl.psnc.vlab.util.i18n.ResourceBundleManager;

/**
 * {@link DefaultLongTask} class - DefaultLongTask - abstract class is used as a
 * base class for time consuming operations. Some of the utility methods have
 * been already implemented. Child classes are required to override
 * doInBackground() method.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class DefaultLongTask extends SwingWorker<Object, String[]> implements ProgressTask {

	/** Default serial uid */
	private static final long serialVersionUID = 1L;

	/** Instance of logger */
	private Log log = LogFactory.getLog(this.getClass().getName());

	/** Store an instance of progress task */
	private ProgressTask progress;

	/** Stores instance of parent class */
	private Component parent;

	/** Stores Action instance */
	private Action action;

	/** Stores please wait message */
	private static String MSG_WAIT;

	/**
	 * Crates a new instance of DefaultLongTask
	 * 
	 * @param parent parent object of this DefaultLongTask
	 * @param progress instance of {@link ProgressTask}
	 * @param action action object for this long task
	 */
	protected DefaultLongTask(Component parent, ProgressTask progress, Action action) {
		this.parent = parent;
		this.progress = progress;
		this.action = action;
		try {
			ResourceBundleManager bundleM = new ResourceBundleManager("bundle/common_gui");
			MSG_WAIT = bundleM.getValue("please.wait");
		} catch (Exception e) {
			log.error("Bundle not found. Details: " + e.getMessage());
			MSG_WAIT = "";
		}
	}

	/**
	 * Crates a new instance of DefaultLongTask
	 * 
	 * @param progress instance of {@link ProgressTask}
	 * @param action action object for this long task
	 */
	protected DefaultLongTask(ProgressTask progress, Action action) {
		this(null, progress, action);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.SwingWorker#process(java.util.List)
	 */
	@Override
	protected void process(List<String[]> arg0) {
		String value[] = arg0 != null ? arg0.get(arg0.size() - 1) : null;
		progress.updateMessage(value);
	}

	/**
	 * Update progress bar. The message is constructed from parent component
	 * title, given message and please wait message
	 * 
	 * @param message message to display
	 */
	protected void updateProgress(String message) {
		publish(new String[] { String
				.format("<html>%s %s %s</html>", getTitle(), message, MSG_WAIT) });
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.SwingWorker#done()
	 */
	@Override
	protected void done() {
		try {
			log.debug("<done>");
			Object result = get();
			if (result == null) {
				return;
			}
			if (result instanceof LongTaskResult) {
				processLongTaskResult((LongTaskResult) result);
			}

		} catch (Exception err) {
			err.printStackTrace();
			JMessage.showErrorMessage(parent, err.getLocalizedMessage());
		} finally {
			if (progress != null) {
				progress.hideProgress();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see pl.psnc.vlab.util.gui.progress.ProgressTask#hideProgress()
	 */
	public void hideProgress() {
		progress.hideProgress();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see pl.psnc.vlab.util.gui.progress.ProgressTask#showProgress(boolean)
	 */
	public void showProgress(boolean interminate) {
		progress.showProgress(interminate);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see pl.psnc.vlab.util.gui.progress.ProgressTask#showProgress(java.lang.String,
	 *      boolean)
	 */
	public void showProgress(String message, boolean interminate) {
		progress.showProgress(message, interminate);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see pl.psnc.vlab.util.gui.progress.ProgressTask#updateMessage(java.lang.String[])
	 */
	public void updateMessage(String... messages) {
		progress.updateMessage(messages);
	}

	/**
	 * Show proceed question message for this LongTask. The option pane with YES
	 * and NO choices is shown. User can choose whether to proceed or quit.
	 * 
	 * @param message proceed question
	 * @return <code>JMessage.YES_OPTION</code> if proceed,
	 *         <code>JMessage.NO_OPTION</code> otherwise
	 */
	protected int showProceedQuestion(String message) {
		int result = JMessage.showConfirmQuestionMessage(parent, message, JMessage.YES_NO_OPTION);
		if (result == JMessage.NO_OPTION) {
			action.setEnabled(true);
		}
		return result;
	}

	// -------------------------------------------------------------------------
	// ---- Setters / Getters

	/**
	 * Get value of action
	 * 
	 * @return the action
	 */
	public Action getAction() {
		return action;
	}

	// -------------------------------------------------------------------------
	// ---- Private Helper methods

	/**
	 * Process the result as LongTaskResult
	 * 
	 * @param result instance of {@link LongTaskResult}
	 */
	private synchronized void processLongTaskResult(LongTaskResult result) {
		if (result.getFocusableComponent() != null) {
			result.getFocusableComponent().requestFocus();
		}
		String msg = result.getMessage();
		if (msg != null) {
			JMessage.showInfoMessage(parent, msg);
		}

	}

	/**
	 * Get parent title. If parent is NULL, empty string is returned.
	 * 
	 * @return parent frame title, or empty string when parent is NULL.
	 */
	// TODO add frame title
	private String getTitle() {
		if (parent == null) {
			return "";
		}
		return String.format("[<i>%s</i>]", "");
	}
}
