/*
 * ActionFactory.java
 *
 * Created on 14 maj 2004, 15:29
 */

package pl.psnc.vlab.util.gui.action.factory;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.gui.action.AbstractActionDefault;
import pl.psnc.vlab.util.gui.action.LocalizedObservableAbstractAction;

/**
 * {@link ActionFactory} class - stores system actions. The action search
 * methods are also available.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class ActionFactory {

	/** Instance of logger */
	private static Log log = LogFactory.getLog(ActionFactory.class);

	/** Stores instance of action map */
	private static Map<ActionType, AbstractActionDefault> actionMap = new HashMap<ActionType, AbstractActionDefault>();

	/**
	 * Creates a new instance of {@link ActionFactory}
	 */
	private ActionFactory() {
	}

	/**
	 * Get action by the given action type
	 * 
	 * @param type type of the action to retrieve
	 * @return instance of {@link AbstractActionDefault}
	 */
	public static AbstractActionDefault getAction(ActionType type) {
		AbstractActionDefault action = actionMap.get(type);
		if (action == null) {
			action = newInstance(type.getType());
			if (action != null) {
				putAction(type, action);
			}
		}
		return action;
	}

	/**
	 * Get observable interface from the given action
	 * 
	 * @param type type of the action to retrieve
	 * @return instance of {@link Observable} or NULL if not Observable
	 */
	public static Observable getObservable(ActionType type) {
		AbstractActionDefault action = getAction(type);
		if (action instanceof LocalizedObservableAbstractAction) {
			return ((LocalizedObservableAbstractAction) action).getObservable();
		}
		return null;
	}

	/**
	 * Get observer interface from the given action
	 * 
	 * @param type type of the action to retrieve
	 * @return instance of {@link Observer} or NULL if not Observer
	 */
	public static Observer getObserver(ActionType type) {
		AbstractActionDefault action = getAction(type);
		if (action instanceof Observer) {
			return (Observer) action;
		}
		return null;
	}

	/**
	 * Store an action on the action list
	 * 
	 * @param type action type, instance of {@link ActionType}
	 * @param action action object, instance of {@link AbstractActionDefault}
	 */
	public static void putAction(ActionType type, AbstractActionDefault action) {
		log.debug("<putAction:" + type + ">");
		if (action == null) {
			log.warn("Action is NULL");
			return;
		}
		actionMap.put(type, action);
	}

	// --------------------------------------------------------------------------
	// ---- Private helper classes

	/**
	 * Creates a new instance of action for the given action type.
	 * 
	 * @param <T> instance of {@link AbstractActionDefault}
	 * @param type type of the action object to be created
	 * @return new instance of {@link AbstractActionDefault}
	 * 
	 */
	private static <T extends AbstractActionDefault> AbstractActionDefault newInstance(Class<T> type) {
		try {
			Constructor<T> actionConst = type.getConstructor();
			return actionConst.newInstance();
		} catch (Exception e) {
			log.error(e.getMessage());
			return null;
		}

	}
}
