/*
 * StringTools.java
 *
 * Created on 20 kwiecie� 2004, 12:48
 */

package pl.psnc.vlab.util;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;

/**
 * {@link FileTools} class - set of utility methods on files.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class FileTools {

	/** System line separator I.e. "\n" in Linux systems */
	public static String LINE_SEPARATOR = System.getProperty("line.separator");

	/** System file separator. I.e. "/" in Linux systems */
	public static String FILE_SEPARATOR = System.getProperty("file.separator");

	/** User working directory */
	public static String WORKING_DIRECTORY = System.getProperty("user.dir");

	/** User home directory */
	public static String HOME_DIRECTORY = System.getProperty("user.home");

	/** Directory containing images */
	public static String IMAGE_DIRECTORY = "images/";

	// tbd
	public final static String jpeg = "jpeg";
	public final static String jpg = "jpg";
	public final static String gif = "gif";
	public final static String tiff = "tiff";
	public final static String tif = "tif";
	public final static String png = "png";
	public final static String jgx = "jgx";

	/**
	 * Get the extension of a file.
	 * 
	 * @param f the file to be checked
	 * @return file extension
	 */
	public static String getExtension(File f) {
		String ext = null;
		String s = f.getName();
		int i = s.lastIndexOf('.');

		if (i > 0 && i < s.length() - 1) {
			ext = s.substring(i + 1).toLowerCase();
		}
		return ext;
	}

	/**
	 * Get the extension of a file by the file name
	 * 
	 * @param fileName file name
	 * @return file extension
	 */
	public static String getExtension(String fileName) {
		String ext = null;
		int i = fileName.lastIndexOf('.');

		if (i > 0 && i < fileName.length() - 1) {
			ext = fileName.substring(i + 1).toLowerCase();
		}
		return ext;
	}

	/**
	 * Get the file name without extension
	 * 
	 * @param fileName name of the file
	 * @return file name without extension
	 */
	public static String getBase(String fileName) {
		String base = fileName;
		int i = fileName.lastIndexOf('.');

		if (i > 0 && i < fileName.length() - 1) {
			base = fileName.substring(0, i);
		}
		return base;
	}

	/**
	 * Get the url without extension
	 * 
	 * @return url url to file without the file extension
	 * @param url url to be converted
	 */
	public static String getBase(URL url) {
		return getBase(url.toString());
	}

	/**
	 * Check whether the specyfied file by this URL exists.
	 * 
	 * @return <CODE>true</CODE> if the file exists, <CODE>false</CODE>
	 *         otherwise
	 * @param file the File representation of the given file
	 */
	public static boolean fileExists(File file) {
		return file == null ? false : file.exists();
	}

	/**
	 * Check whether the specyfied file by this URL exists.
	 * 
	 * @param fileName the file name
	 * @return <CODE>true</CODE> if the file exists, <CODE>false</CODE>
	 *         otherwise
	 */
	public static boolean fileExists(String fileName) {
		return fileName == null ? false : new File(fileName).exists();
	}

	/**
	 * Returns the classname without the package. Example: If the input class is
	 * java.lang.String than the return value is String.
	 * 
	 * @param cl The class to inspect
	 * @return The classname
	 * 
	 */
	public static String getClassNameWithoutPackage(Class<?> cl) {
		String className = cl.getName();
		int pos = className.lastIndexOf('.') + 1;
		if (pos == -1) {
			pos = 0;
		}
		return className.substring(pos);
	}

	// tbd refactoring
	/**
	 * Get the resource name from its absolute path.
	 * 
	 * @param resAbsolutePath resource absolute path
	 * @return resource name from its absolute path.
	 */
	public static String getResourceName(String resAbsolutePath) {
		int i = resAbsolutePath.lastIndexOf("jar!");
		if (i == -1) {
			return resAbsolutePath;
		}
		if (resAbsolutePath.charAt(i + 4) == '/') {
			return resAbsolutePath.substring(i + 5, resAbsolutePath.length());
		}
		return resAbsolutePath.substring(i + 4, resAbsolutePath.length());
	}

	/**
	 * Reads the content of the given input stream into a byte array
	 * 
	 * @param inputStream input stream ({@link InputStream})
	 * @return content of the given input stream
	 */
	public static byte[] getContent(InputStream inputStream) {
		if (inputStream == null) {
			return null;
		}
		BufferedReader bufferedReader = null;
		try {
			bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
			String line = null;
			StringBuffer result = new StringBuffer();
			while ((line = bufferedReader.readLine()) != null) {
				result.append(line);
				result.append(LINE_SEPARATOR);
			}
			return result.toString().getBytes();
		} catch (Exception e) {
			return null;
		} finally {
			closeIgnoringException(bufferedReader);
		}
	}

	/**
	 * Reads the content of the given file into a byte array
	 * 
	 * @param file current file
	 * @return byte representation of the file content or null if error occurs
	 */
	public static byte[] getContent(File file) {
		InputStream inputStream;
		try {
			inputStream = new FileInputStream(file);
		} catch (FileNotFoundException e) {
			return null;
		}
		return getContent(inputStream);
	}

	/**
	 * Close the given stream ignoring exceptions thrown
	 * 
	 * @param c instance of {@link Closeable} interface
	 */
	private static void closeIgnoringException(Closeable c) {
		if (c == null) {
			return;
		}
		try {
			c.close();
		} catch (IOException e) {
			// ok to ignore
		}
	}

}
