/*
 * DateTools.java
 *
 * Created on 10 grudzie� 2002, 21:33
 */

package pl.psnc.vlab.util;

import java.util.*;
import java.text.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * {@link DateTools} class - utility class for dealing with dates.
 *
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 *
 */
public class DateTools {
    
    /** Instance of logger */
    private Log log = LogFactory.getLog(this.getClass().getName());
    
    private  Locale locale = null;
    private DateFormatSymbols dateFormatSymbols = null;
    private SimpleDateFormat dateFormat = null;
    private SimpleDateFormat timeFormat = null;
    private SimpleDateFormat dateTimeFormat = null;
    
    /** Time format*/
    private static String TIME_FORMAT = "kk:mm:ss";
    /** Date format*/
    private static String DATE_FORMAT = "dd, MMMM yyyy";
    /** Date and time  format*/
    private static String DATE_TIME_FORMAT = DATE_FORMAT +" "+ TIME_FORMAT;
    
    
    
    /**
     * Creates new instance of Date Tools with default locale (English),
     * default date pattern (dd, MMMM yyyy) and default time pattern (kk:mm:ss a)
     */
    public DateTools() {
        this.locale = Locale.ENGLISH;
        this.dateFormat = new SimpleDateFormat(DATE_FORMAT, this.dateFormatSymbols);
        this.timeFormat = new SimpleDateFormat(TIME_FORMAT, this.dateFormatSymbols);
        this.dateTimeFormat = new SimpleDateFormat(DATE_TIME_FORMAT, this.dateFormatSymbols);
    }
    
    
    /**
     * Creates new instance of Date Tools
     * @param datePattern date pattern
     * @param timePattern time pattern
     * @param locale locale, which will be used for the given date object
     */
    public DateTools(Locale locale, String datePattern, String timePattern ) {
        this.locale = locale != null ? locale : Locale.ENGLISH;
        this.dateFormatSymbols = new DateFormatSymbols(this.locale);
        String dataAndTime = null;
        if(datePattern == null){
            this.dateFormat = new SimpleDateFormat(DATE_FORMAT, this.dateFormatSymbols);
            dataAndTime = DATE_FORMAT;
        } else {
            this.dateFormat = new SimpleDateFormat(datePattern, this.dateFormatSymbols);
            dataAndTime = datePattern;
        }
        
        if(timePattern == null){
            this.timeFormat = new SimpleDateFormat(TIME_FORMAT, this.dateFormatSymbols);
            dataAndTime = new String(dataAndTime +" "+TIME_FORMAT); 
        } else {
            this.timeFormat = new SimpleDateFormat(timePattern, this.dateFormatSymbols);
            dataAndTime = new String(dataAndTime +" "+timePattern);
        }
        
        
        this.dateTimeFormat = new SimpleDateFormat(dataAndTime, this.dateFormatSymbols);
    }
    
    
    
    
    /**
     * Converts a given date into a string representation according to
     * locale and date pattern.
     * @return date in the string representaion
     * @param date date
     */
    public String getDateAsString(Date date){
        return this.dateFormat.format(date);
    }
    
    /**
     * Converts a given date into a string time representation according to
     *  locale and time pattern.
     * @return date in the string representaion
     * @param date time in the string representaion
     */
    public String getTimeAsString(Date date) {
        return this.timeFormat.format(date);
        
    }
    
    
    /**
     * Converts a given date into a string date and time representation according to
     *  locale and time pattern.
     * @return date in the string representaion
     * @param date time in the string representaion
     */
    public String getDateTimeAsString(Date date) {
        return this.dateFormat.format(date) + " " + this.timeFormat.format(date);
        
    }
    
    
    
    /**
     * Get date and time from string value.
     * @param value date object
     * @return new date object or NULL if the value is incorrect date
     */
    public Date getDateTimeFromString(String value){
        try{
            return dateTimeFormat.parse(value);
        }catch(java.text.ParseException pE){
            return null;
        }
    }
    
    
    
    /**
     * Get date from string value.
     * @param value date object
     * @return new date object or NULL if the value is incorrect date
     */
    public Date getDateFromString(String value){
        try{
            return dateFormat.parse(value);
        }catch(java.text.ParseException pE){
            return null;
        }
    }
    
    /**
     * Get time from string value.
     * @param value date object
     * @return new date object or NULL if the value is incorrect date
     */
    public Date getTimeFromString(String value){
        try{
            return timeFormat.parse(value);
        }catch(java.text.ParseException pE){
            return null;
        }
    }
    
    
    /**
     * Convert the date and time to the given date and time format
     * @param newFormat new date format
     * @param date date
     * @param time time
     * @return date in the string representaion
     */
    public String convertToFormat(String newFormat, String date, String time){
        log.debug("<convertToFormat:"+newFormat+"|"+date+"|"+time);
        Date dateAsDate = !FormTools.isValidateFormStringNull(date) ? getDateFromString(date) : new Date();
        Date timeAsDate = !FormTools.isValidateFormStringNull(time) ? getTimeFromString(time) : new Date();
        //set time
        dateAsDate = setTime(dateAsDate, timeAsDate);
        SimpleDateFormat newDateFormat = new SimpleDateFormat(newFormat, this.dateFormatSymbols);
        return newDateFormat.format(dateAsDate);
    }
    
    /**
     * Checks whether the given string date representation is valid according
     * to the locale and date pattern.
     * @param date date
     * @return <CODE>true</CODE> if date is valid according to the given locale and format
     * <CODE>false</CODE> otherwise
     */
    public boolean isDateValid(String date){
        try{
            dateFormat.parse(date);
            return true;
        }catch(java.text.ParseException pE){
            return false;
        }
    }
    
    
    
    /**
     * Checks whether the given string time representation is valid according
     * to the locale and time pattern.
     * @return <CODE>true</CODE> if time is valid according to the given locale and format
     * <CODE>false</CODE> otherwise
     * @param time time
     */
    public boolean isTimeValid(String time){
        try{
            timeFormat.parse(time);
            return true;
        }catch(java.text.ParseException pE){
            return false;
        }
    }
    
    
    
    
    
    //-------------------------------------
    
    /**
     * Adds one day to current date
     * @param date date
     * @return date object
     */
    public  Date addDay(Date date) {
        Calendar calendar = Calendar.getInstance(this.locale);
        calendar.setTime(date);
        //ads one week
        calendar.add(Calendar.DATE, 1);
        return calendar.getTime();
    }
    
    
    /**
     * Adds some months to the given date
     * @return current date updated with one day
     * @param months number of month to be added
     * @param date current date
     */
    public  Date addMonth(Date date, int months) {
        Calendar calendar = Calendar.getInstance(this.locale);
        calendar.setTime(date);
        //ads one week
        calendar.add(Calendar.MONTH, months);
        return calendar.getTime();
    }
    
    
    
    /**
     * Adds the specified amount of units to the given date
     * @return new date updated with the specified amount
     * @param field the date field, which should be changed i.e. Calendar.MONTH, Calnedar.MINUTE
     * @param amount amount of units to be added
     * @param date current date
     */
    public  Date add(Date date, int field, int amount) {
        Calendar calendar = Calendar.getInstance(this.locale);
        calendar.setTime(date);
        //ads one week
        calendar.add(field, amount);
        return calendar.getTime();
    }
    
    //-------------------------------------
    
    /**
     * Subtracts one day to current date
     * @param date date
     * @return date object
     */
    public  Date subtractDay(Date date) {
        Calendar calendar = Calendar.getInstance(this.locale);
        calendar.setTime(date);
        //ads one week
        calendar.add(Calendar.DATE, -1);
        return calendar.getTime();
    }
    
    
    
    //------------------------------------------------------
    
    /**
     * Compares two dates objects. If first argument is after the second date return true.
     * @param d1 first date
     * @param d2 second date
     * @return <CODE>true</CODE> if date d1 is greater than d2 <CODE>false</CODE> otherwise
     */
    public boolean isGreater(Date d1, Date d2) {
        if(d1.compareTo(d2) == 1) {
            return true;
        } else {
            return false;
        }
    }
    
    //------------------------------------------------------
    
    /**
     * Compares two dates objects. If first argument is the same as the second date return true.
     * @param d1 date
     * @param d2 date
     * @return <CODE>true</CODE> if date d1 equals to d2 <CODE>false</CODE> otherwise
     */
    public boolean isEqual(Date d1, Date d2) {
        if(d1.compareTo(d2) == 0) {
            return true;
        } else {
            return false;
        }
    }
    
    //------------------------------------------------------
    
    /**
     * Compares two dates objects. If first argument is before the second date return true.
     * @param d1 first date
     * @param d2 second date
     * @return <CODE>true</CODE> if date d1 is smaller than d2 <CODE>false</CODE> otherwise
     */
    public boolean isSmaller(Date d1, Date d2) {
        if(d1.compareTo(d2) == -1) {
            return true;
        } else {
            return false;
        }
    }
    
    
    /**
     * Set the time to the current date
     * @param date date to be set
     * @param time time to be set
     * @return date object
     */
    public Date setTime(Date date, Date time){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        Calendar timeCal= Calendar.getInstance();
        timeCal.setTime(time);
        //set time
        calendar.set(Calendar.HOUR, timeCal.get(Calendar.HOUR));        //set hour
        calendar.set(Calendar.MINUTE, timeCal.get(Calendar.MINUTE));    //set minute
        calendar.set(Calendar.SECOND, timeCal.get(Calendar.SECOND));    //set second
        calendar.set(Calendar.AM_PM, timeCal.get(Calendar.AM_PM));      //set am/pm
        return calendar.getTime();
    }
    
    
    
    
}
