/*
 * DatagramHelper.java
 *
 * Created on 3 czerwiec 2004, 11:28
 */

package pl.psnc.vlab.remote.util;

import java.nio.ByteOrder;
import pl.psnc.vlab.exception.VlabException;
import pl.psnc.vlab.management.session.SessionFactory;
import pl.psnc.vlab.remote.Dgram;
import pl.psnc.vlab.values.management.SessionValue;

/**
 * The helper class used to fill out the fields in the
 * Datagram which is used to communicate with the VLab system
 * @author  <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email  osa@man.poznan.pl
 */
public class DatagramHelper {
    
    /** Specyfies the number of datagrams sent */
    private static int datagramCount = 0;
    
    /**
     * Fills all the fileds in the Datagram object which can be set.
     * The one which are not used are set to default values. Some of the fields
     * which are alterd: new datagram identyfier is generated,
     * user id is set, the group (lab) id, task id (which is unknown) and
     * content id ( unused )
     * @return datagram with the common fields set
     * @param datagramIn datagram (Dgram class)
     * @throws VlabException an exception which may be thrown
     */
     public static Dgram setCommonFields(Dgram datagramIn) throws VlabException{
        //generate datagram id
        datagramIn.setDatagramId( generateDatagramId() );
        SessionValue sessionValue = SessionFactory.getSessionValue();
        //set the user id
        datagramIn.setUserId(sessionValue.getUserId() );
        //set user certificate id
        datagramIn.setUserCertId(sessionValue.getUserCertId());
        //set the lab id
        datagramIn.setGroupId(sessionValue.getLabId());
        //set the task id
        datagramIn.setValidTaskId(0);
        //set the content id
        datagramIn.setContentId(1);
        //set operationStatus id
        datagramIn.setOperationStatusId( VlabRemoteKeys.OPERATION_STATUS_CALL);
        //set inet address
        datagramIn.setInetAddress("localhost");
        //set datgram number
        datagramIn.setDatagramNumber(1);
        //set datagram count
        datagramIn.setDatagramCount(1);
        return datagramIn;
    }
    
     
     /**
      * Fills all the fileds in the Datagram object which can be set.
      * The one which are not used are set to default values. Some of the fields
      * which are alterd: new datagram identifier is generated,
      * user id is set to -1, the group (lab) id is set to -1, task id (which is unknown) and
      * content id ( unused )
      * @return datagram with the common fields set (except the session info).
      * This is DEBUG method
      * @param datagramIn datagram (Dgram class)
      * @throws VlabException an exception which may be thrown
      */
     public static Dgram setCommonFieldsWithoutSession(Dgram datagramIn) throws VlabException{
        //generate datagram id
        datagramIn.setDatagramId( generateDatagramId() );        
        //set the user id
        datagramIn.setUserId(-1 );
        //set user certificate id
        datagramIn.setUserCertId("No user cert id");
        //set the lab id
        datagramIn.setGroupId(-1);
        //set the task id
        datagramIn.setValidTaskId(0);
        //set the content id
        datagramIn.setContentId(1);
        //set operationStatus id
        datagramIn.setOperationStatusId( VlabRemoteKeys.OPERATION_STATUS_CALL);
        //set inet address
        datagramIn.setInetAddress("localhost");
        //set datgram number
        datagramIn.setDatagramNumber(1);
        //set datagram count
        datagramIn.setDatagramCount(1);
        return datagramIn;
    }
    
    /**
     * Generates the datagram identyfier. The following rule
     * is being followed to generate id.
     * The first two digits are reserved for the sender identyfier. This is
     * the id of the module which sends the datagram. The identyfiers can
     * be found at VlabRemoteKeys.
     *
     * The rest seven digits - the next number of the datagram
     *
     * @return
     */
    private static int generateDatagramId(){
        return new Integer(VlabRemoteKeys.VL_SCEN_SUB_ID * 10000000 + datagramCount++ ).intValue();
    }
    
    
    /**
     * Decodes the byte order by the given integer value. There 
     * are two values supported: 
     *  - 0     little endian
     *  - 1     big endian
     * @return byte order
     * @param byteOrder 
     * @throws pl.psnc.vlab.exception.VlabException throws an exception if invalid
     * byte order is set
     */
    public static ByteOrder getByteOrder(int byteOrder) throws VlabException{
        if(byteOrder == VlabRemoteKeys.LITTLE_ENDIAN ){
            return ByteOrder.LITTLE_ENDIAN;
        } else if(byteOrder == VlabRemoteKeys.BIG_ENDIAN){
            return ByteOrder.BIG_ENDIAN;
        }
        throw new VlabException("Internal error: invalid byte order. Got :"+byteOrder);
        
    }
    
    
}


