/*
 * BytesConversion.java
 *
 * Created on 18 marzec 2004, 14:44
 */

package pl.psnc.vlab.remote.util;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.exception.VlabException;
import pl.psnc.vlab.remote.Dgram;
import pl.psnc.vlab.remote.value.ShortMessageValue;
import pl.psnc.vlab.util.FormTools;
import pl.psnc.vlab.values.management.SessionValue;

/**
 * {@link BytesConversion} class - helper class is used to decode or encode the
 * data which is sent between the application and the VLab System
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class BytesConversion {

	/** The size (in bytes) of the integer variable */
	protected static int INT_SIZE = 4;

	/** Logger instance */
	private Log log = LogFactory.getLog(this.getClass().getName());

	/**
	 * Converts a stream of bytes to the SessionValue object. The given stream
	 * of bytes should represents the following c++ structure : struct
	 * getAppSessionByKeyAnswer { unsigned int appsId; //session id (4 bytes)
	 * char apps_key[35]; //session key char apps_lang[10]; //session language
	 * char apps_country[10]; //session country double apps_timestamp; //session
	 * timestamp int apps_timelimit; //session timelimit unsigned int userId;
	 * //session owner's id char userCertId[150]; //session owner's id unsigned
	 * int labId; //session owner's lab char login[35]; //session owner's login
	 * name };
	 * 
	 * @return session value object
	 * @param datagramOut
	 * @throws pl.psnc.vlab.exception.VlabException
	 */
	public SessionValue convertToSessionValue(Dgram datagramOut) throws VlabException {
		byte[] sessionDetails = datagramOut.getBinaryContent();
		if (sessionDetails == null || sessionDetails.length == 0) {
			log.debug("Bad session details - NULL");
			return null;
		}
		ByteBuffer byteBuffer = ByteBuffer.wrap(sessionDetails);
		ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
		byteBuffer.order(byteOrder);
		SessionValue sessionValue = new SessionValue();

		sessionValue.setSessionId(byteBuffer.getInt()); // set session id

		// skip the session Key
		byte[] tmpArr = new byte[35];
		byteBuffer.get(tmpArr, 0, 35);

		tmpArr = new byte[10];
		byteBuffer.get(tmpArr, 0, 10);
		sessionValue.setLanguageCode(getString(tmpArr)); // set language

		tmpArr = new byte[10];
		byteBuffer.get(tmpArr, 0, 10);
		sessionValue.setCountryCode(getString(tmpArr)); // set country

		sessionValue.setTimeLimit(byteBuffer.getInt()); // set time limit

		sessionValue.setUserId(byteBuffer.getInt()); // set user id

		tmpArr = new byte[150];
		byteBuffer.get(tmpArr, 0, 150);
		sessionValue.setUserCertId(getString(tmpArr)); // set user certificate
		// id

		sessionValue.setLabId(byteBuffer.getInt()); // set lab id

		tmpArr = new byte[35];
		byteBuffer.get(tmpArr, 0, 35); // set user name
		sessionValue.setUserName(getString(tmpArr));
		return sessionValue;
	}

	/**
	 * Converts a stream of bytes to boolean type struct isAppSessionValidAnswer {
	 * int isValid; //boolean is valid (0/1) };
	 * 
	 * @return <CODE>true</CODE> session is valid <CODE>false</CODE>
	 *         otherwise
	 * @param datagramOut
	 * @throws pl.psnc.vlab.exception.VlabException
	 */
	public boolean isSessionValidAnswer(Dgram datagramOut) throws VlabException {
		byte[] sessionDetails = datagramOut.getBinaryContent();
		if (sessionDetails == null || sessionDetails.length == 0) {
			log.debug("Bad session details - NULL");
			return false;
		}
		ByteBuffer byteBuffer = ByteBuffer.wrap(sessionDetails);
		ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
		byteBuffer.order(byteOrder);
		int res = byteBuffer.getInt();
		log.debug("result-" + res);
		return res == 0 ? false : true;
	}

	/**
	 * Converts a string value into fixed size byte array. The UTF-8 encoding is
	 * used.
	 * 
	 * @return array of bytes, which represent the given string value
	 * @param value string value to be converted
	 * @param size the length of the result array
	 */
	public byte[] toBytes(String value, int size) {
		if (FormTools.isValidateFormStringNull(value)) {
			log.debug("Value is Null");
			byte[] res = new byte[size];
			Arrays.fill(res, (byte) 0);
			return res;
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(size);

		byte b[] = null;
		try {
			b = value.getBytes("UTF-8");
		} catch (UnsupportedEncodingException e) {
			b = value.getBytes();
		}
		if (b.length > size) {
			log.debug("Value is greater than size. Truncate");
			return byteBuffer.put(b, 0, size).array();
		}
		byteBuffer.put(b);
		byte[] empty = new byte[size - b.length];
		Arrays.fill(empty, (byte) 0);
		byteBuffer.put(empty);
		return byteBuffer.array();
	}

	/**
	 * Encodes the ShortMessage struct into array of bytes: struct shortMessage {
	 * char buffer[256]; int len; };
	 * 
	 * @return an object representation of thie shortMessage stucture
	 * @param shortMsg
	 */
	public byte[] encodeShortMessage(ShortMessageValue shortMsg) {
		if (shortMsg == null) {
			log.debug("short msg is Null");
			return null;
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(shortMsg.getStructureLength() + INT_SIZE);
		// encode data
		byteBuffer.putInt(shortMsg.getStructureLength());
		byteBuffer.put(toBytes(shortMsg.getMessage(), shortMsg.getStructureLength()));
		return byteBuffer.array();
	}

	/**
	 * Decodes the ShortMessage struct: struct shortMessage { char buffer[256];
	 * int len; };
	 * 
	 * @param binaryData byte array, which represents the VLab shortMessage
	 *            structure (The structure was presented in the method
	 *            description)
	 * @return an object representation of thie shortMessage stucture
	 */
	public ShortMessageValue getShortMessage(byte[] binaryData, ByteOrder byteOrder) {
		if (binaryData == null || binaryData.length == 0) {
			log.debug("binaryData is Null");
			return null;
		}
		ByteBuffer byteBuffer = ByteBuffer.wrap(binaryData);
		byteBuffer.order(byteOrder);
		ShortMessageValue shortMessageValue = new ShortMessageValue();
		shortMessageValue.setStructureLength(byteBuffer.getInt());
		log.debug("Legth-" + shortMessageValue.getStructureLength());
		shortMessageValue.setMessage(new String(binaryData, 4, shortMessageValue
				.getStructureLength()));

		return shortMessageValue;
	}

	/**
	 * Decodes an array of byte to the string. It reads the byte array until 0
	 * byte is found or the array ends. The UTF-8 encoding is used.
	 * 
	 * @param binaryData encoded string as an byte array
	 * @return the string value represented by this array.
	 */
	public String getString(byte[] binaryData) {
		if (binaryData == null || binaryData.length == 0) {
			log.debug("binaryData is Null");
			return null;
		}
		int size = binaryData.length;
		int counter = 0;
		for (counter = 0; counter < size; counter++) {
			if (binaryData[counter] == 0) {
				break;
			}
		}// end of for

		return new String(binaryData, 0, counter);
	}
}
