/*
 * VlabRemoteImpl.java
 *
 * Created on 9 marzec 2004, 13:23
 */

package pl.psnc.vlab.remote.session;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.exception.VlabException;
import pl.psnc.vlab.remote.Dgram;
import pl.psnc.vlab.remote.VlabdgramLocator;
import pl.psnc.vlab.remote.VlabdgramPortType;
import pl.psnc.vlab.remote.util.BytesConversion;
import pl.psnc.vlab.remote.util.DatagramHelper;
import pl.psnc.vlab.remote.util.VlabRemoteKeys;
import pl.psnc.vlab.remote.value.ShortMessageValue;
import pl.psnc.vlab.util.FormTools;
import pl.psnc.vlab.values.management.SessionValue;

/**
 * {@link VlabSessionRemoteImpl} class - implementation of the interface to the
 * VLab Communication Proxy module (which is written in C). It allows sending
 * messages to the VLab sytem.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class VlabSessionRemoteImpl implements VlabSessionRemote {

	/** Logger instance */
	private Log log = LogFactory.getLog(this.getClass().getName());

	/** The service port type */
	private VlabdgramPortType port;

	public VlabSessionRemoteImpl() throws VlabException {
		try {
			port = new VlabdgramLocator().getvlabdgram();
		} catch (Exception e) {
			throw new VlabException(e.getMessage(), e);
		}
	}

	/**
	 * Retrieve session information by the given session key
	 * 
	 * @return session object
	 * @param sessionKey session key
	 * @throws VlabException VLabException indicates an exception which may
	 *             occur during call
	 */
	public SessionValue getAppSession(String sessionKey) throws VlabException {
		try {
			if (FormTools.isValidateFormStringNull(sessionKey)) {
				log.debug("Session Key is null");
				return null;
			}
			Dgram datagramIn = new Dgram();
			datagramIn = DatagramHelper.setCommonFieldsWithoutSession(datagramIn);
			// set operation id
			datagramIn.setOperationId(VlabRemoteKeys.MONITORING_GET_SESSION);
			// set binary content
			byte[] dataArray = new BytesConversion().toBytes(sessionKey, 35); // convert
			// session
			// key
			// to
			// 35
			// bytes
			// arr

			datagramIn.setBinaryContent(dataArray);
			// set content size
			datagramIn.setContentSize(dataArray.length);
			// datagramIn
			Dgram datagramOut = port.callCommproxySyncApp(datagramIn);

			if (datagramOut.getOperationStatusId() == VlabRemoteKeys.OPERATION_STATUS_ERROR) {
				ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
				ShortMessageValue shortMessageV = new BytesConversion().getShortMessage(datagramOut
						.getBinaryContent(), byteOrder);
				log.debug("Error during call. Details:" + shortMessageV.getMessage());
				throw new VlabException(shortMessageV.getMessage());
			}

			return new BytesConversion().convertToSessionValue(datagramOut);
		} catch (Exception e) {
			e.printStackTrace();
			throw new VlabException(e.getMessage());
		}
	}

	/**
	 * Checks whether the current session is valid. The session is valid if the
	 * following condition is true: current_timestamp < last_valid_timestamp +
	 * timestamp_limit timestamo_limit is stored in the database
	 * 
	 * @return <CODE>true</CODE> if the session is valid <CODE>false</CODE>
	 *         otherwise
	 * @param sessionKey session key
	 * @throws VlabException VLabException indicates an exception which may
	 *             occur during call
	 */
	public boolean isSessionValid(String sessionKey) throws VlabException {
		try {
			if (FormTools.isValidateFormStringNull(sessionKey)) {
				log.debug("Session Key is null");
				return false;
			}
			Dgram datagramIn = new Dgram();
			datagramIn = DatagramHelper.setCommonFields(datagramIn);
			// set operation id
			datagramIn.setOperationId(VlabRemoteKeys.MONITORING_IS_SESSION_VALID);
			// set binary content
			byte[] dataArray = new BytesConversion().toBytes(sessionKey, 35); // convert
			// session
			// key
			// to
			// 35
			// bytes
			// arr
			datagramIn.setBinaryContent(dataArray);
			// set content size
			datagramIn.setContentSize(dataArray.length);
			Dgram datagramOut = port.callCommproxySyncApp(datagramIn);
			if (datagramOut.getOperationStatusId() == VlabRemoteKeys.OPERATION_STATUS_ERROR) {
				ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
				ShortMessageValue shortMessageV = new BytesConversion().getShortMessage(datagramOut
						.getBinaryContent(), byteOrder);
				log.debug("Error during call. Details:" + shortMessageV.getMessage());

				return false;
			}

			return new BytesConversion().isSessionValidAnswer(datagramOut);
		} catch (Exception e) {
			throw new VlabException(e.getMessage());
		}
	}

	/**
	 * Deletes session from the database by the given session key
	 * 
	 * @param sessionKey session key
	 * @throws VlabException VLabException indicates an exception which may
	 *             occur during call
	 * @return <CODE>true</CODE> if the operation succeeds <CODE>false</CODE>
	 *         otherwise
	 */
	public boolean removeSession(String sessionKey) throws VlabException {
		try {

			if (FormTools.isValidateFormStringNull(sessionKey)) {
				log.debug("Session Key is null");
				return false;
			}

			// testing purpose code
			if (sessionKey.startsWith("test")) {
				return true;
			}

			Dgram datagramIn = new Dgram();
			datagramIn = DatagramHelper.setCommonFieldsWithoutSession(datagramIn);
			// set operation id
			datagramIn.setOperationId(VlabRemoteKeys.MONITORING_REMOVE_SESSION);
			// set binary content
			byte[] dataArray = new BytesConversion().toBytes(sessionKey, 35); // convert
			// session
			// key
			// to
			// 35
			// bytes
			// arr
			datagramIn.setBinaryContent(dataArray);
			// set content size
			datagramIn.setContentSize(dataArray.length);
			Dgram datagramOut = port.callCommproxySyncApp(datagramIn);
			ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
			ShortMessageValue shortMessageV = new BytesConversion().getShortMessage(datagramOut
					.getBinaryContent(), byteOrder);
			if (datagramOut.getOperationStatusId() == VlabRemoteKeys.OPERATION_STATUS_ERROR) {
				log.debug("Error during call. Details:" + shortMessageV.getMessage());
				return false;
			}
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			throw new VlabException(e.getMessage());
		}
	}

	/**
	 * Updates timestamp in the current session
	 * 
	 * @param sessionKey session key
	 * @throws VlabException VLabException indicates an exception which may
	 *             occur during call
	 * @return <CODE>true</CODE> if the operation succeeds <CODE>false</CODE>
	 *         otherwise
	 */
	public boolean updateSession(String sessionKey) throws VlabException {
		try {
			if (FormTools.isValidateFormStringNull(sessionKey)) {
				log.debug("Session Key is null");
				return false;
			}

			Dgram datagramIn = new Dgram();
			datagramIn = DatagramHelper.setCommonFields(datagramIn);
			// set operation id
			datagramIn.setOperationId(VlabRemoteKeys.MONITORING_UPDATE_SESSION);
			// set binary content
			byte[] dataArray = new BytesConversion().toBytes(sessionKey, 35); // convert
			// session
			// key
			// to
			// 35
			// bytes
			// arr
			datagramIn.setBinaryContent(dataArray);
			// set content size
			datagramIn.setContentSize(dataArray.length);
			Dgram datagramOut = port.callCommproxySyncApp(datagramIn);
			ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
			ShortMessageValue shortMessageV = new BytesConversion().getShortMessage(datagramOut
					.getBinaryContent(), byteOrder);
			if (datagramOut.getOperationStatusId() == VlabRemoteKeys.OPERATION_STATUS_ERROR) {
				log.debug("Error during call. Details:" + shortMessageV.getMessage());
				return false;
			}
			return true;
		} catch (Exception e) {
			// e.printStackTrace();
			throw new VlabException(e.getMessage(), e);
		}
	}

	/**
	 * Get the user proxy
	 * 
	 * @return user proxy as a byte array
	 * @param userId
	 * @throws VlabException indicates an exception which may occur during call
	 */
	public byte[] getUserProxy(int userId) throws VlabException {
		log.debug("<getUserProxy: id=" + userId + ">");
		try {

			Dgram datagramIn = new Dgram();
			datagramIn = DatagramHelper.setCommonFieldsWithoutSession(datagramIn);
			// set operation id
			datagramIn.setOperationId(VlabRemoteKeys.MONITORING_GET_USER_PROXY);
			// set binary content
			ByteBuffer byteBuffer = ByteBuffer.allocate(4);
			byteBuffer.putInt(userId);
			byte dataArray[] = byteBuffer.array();
			datagramIn.setBinaryContent(dataArray);
			// set content size
			datagramIn.setContentSize(dataArray.length);
			Dgram datagramOut = port.callCommproxySyncApp(datagramIn);
			if (datagramOut.getOperationStatusId() == VlabRemoteKeys.OPERATION_STATUS_ERROR) {
				ByteOrder byteOrder = DatagramHelper.getByteOrder(datagramOut.getByteOrder());
				ShortMessageValue shortMessageV = new BytesConversion().getShortMessage(datagramOut
						.getBinaryContent(), byteOrder);
				log.debug("Error during call. Details:" + shortMessageV.getMessage());
				throw new VlabException(shortMessageV.getMessage());
			}

			log.debug("</getUserProxy>");
			return datagramOut.getBinaryContent();
		} catch (Exception e) {
			// e.printStackTrace();
			log.debug("</getUserProxy:error>");
			throw new VlabException(e.getMessage(), e);
		}
	}

}
