package pl.psnc.vlab.exception;

import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.vlab.util.i18n.ResourceBundleManager;

/**
 * {@link AbstractException} class - abstract class represents a generic
 * exception
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class AbstractException extends Exception {

	/** Stores an instance of Logger */
	Log log = LogFactory.getLog(getClass().getName());

	/** Default serial version uid */
	static final long serialVersionUID = 1;

	/** Stores an instance of Resource Bundle Manager */
	private static ResourceBundleManager bundleManager;

	/** Stores an error code */
	private int errorCode;

	/**
	 * Creates a new AbstractException with error code specified
	 * 
	 * @param errorCode error code
	 */
	public AbstractException(Integer errorCode) {
		this(errorCode, "");
	}

	/**
	 * Creates a new AbstractException with error code and source exception
	 * specified
	 * 
	 * @param errorCode error code
	 * @param sourceExc source exception, which caused this event
	 */
	public AbstractException(Integer errorCode, Exception sourceExc) {
		this(errorCode, sourceExc != null ? sourceExc.getMessage() : null, sourceExc);
	}

	/**
	 * Creates a new AbstractException with error code and error message
	 * specified
	 * 
	 * @param errorCode error code
	 * @param message error message
	 */
	public AbstractException(Integer errorCode, String message) {
		this(errorCode, message, null);
	}

	/**
	 * Creates a new AbstractException with error code, error message and source
	 * exception specified
	 * 
	 * @param errorCode error code
	 * @param message error message
	 * @param sourceExc source exception, which caused this event
	 */
	public AbstractException(Integer errorCode, String message, Exception sourceExc) {
		super(message, sourceExc);
		this.errorCode = errorCode;
	}

	/**
	 * Get bundle with the i18n keys for the given exception
	 * 
	 * @return bundle name for the given exception
	 */
	public abstract String getBundle();

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Throwable#getLocalizedMessage()
	 */
	@Override
	public String getLocalizedMessage() {
		String locMsg = getMessage();
		try {
			if (bundleManager == null) {
				bundleManager = new ResourceBundleManager(getBundle(), Locale.getDefault());
			}

			if (bundleManager.hasKey(locMsg)) {
				locMsg = bundleManager.getValue(locMsg);
			}
		} catch (Exception e) {
			log.error("Error while initializing bundle:" + e.getMessage());
		}
		return locMsg;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Throabke#printStackTrace()
	 */
	@Override
	public void printStackTrace() {
		Throwable t = getCause();
		if (t != null) {
			t.printStackTrace();
			return;
		}
		super.printStackTrace();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Throabke#printStackTrace()
	 */
	@Override
	public void printStackTrace(java.io.PrintStream printStream) {
		Throwable t = getCause();
		if (t != null) {
			t.printStackTrace(printStream);
			return;
		}
		super.printStackTrace(printStream);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Throabke#printStackTrace()
	 */
	@Override
	public void printStackTrace(java.io.PrintWriter printWriter) {
		Throwable t = getCause();
		if (t != null) {
			t.printStackTrace(printWriter);
			return;
		}
		super.printStackTrace(printWriter);

	}

	// ----------------------------------------------------------------------
	// ---- Setters / Getters

	/**
	 * Returns error code of this exception
	 * 
	 * @return exception error code
	 */
	public int getErrorCode() {
		return this.errorCode;
	}
}
