/* * ConfigurationImpl.java
 *
 * Created on March 8, 2005, 2:24 PM
 */

package pl.psnc.vlab.conf;

import java.io.File;
import java.io.InputStream;
import java.util.Properties;

import pl.psnc.vlab.exception.VlabException;
import pl.psnc.vlab.util.FileTools;

/**
 * {@link ConfigurationImpl} class - implementation of {@link IConfiguration}
 * interface. This class is used for retrieving user custom properties.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public class ConfigurationImpl implements IConfiguration {

	/** Holds the application properties - initialised by default to empty list */
	private Properties properties;

	/**
	 * Creates a new instance of ConfigurationImpl
	 * 
	 * @param appConfig a path to the file, which contains the app configuration
	 *        (as a properties)
	 */
	protected ConfigurationImpl(String appConfig) throws VlabException {
		try {
			InputStream in = ConfigurationImpl.class.getResourceAsStream(appConfig);
			this.properties = new Properties();
			properties.load(in);
		} catch (Exception e) {
			e.printStackTrace();
			throw new VlabException("Error while initializing config: " + appConfig + ". Details: "
					+ e.getMessage(), e);
		}
	}

	/**
	 * Returns the absolute path to the folder, where all the configuration
	 * settings is stored.
	 * 
	 * @return path to the configuration folder
	 */
	public String getSystemConfDir() {
		String result = getProperty(KEY_APP_CONF_DIR);
		if (result == null) {
			result = new StringBuffer().append(FileTools.HOME_DIRECTORY).append(
					FileTools.FILE_SEPARATOR).append(".").append(getProperty(KEY_APP_NAME))
					.toString();
		}
		// check whether dir exists
		if (!FileTools.fileExists(result)) {
			new File(result).mkdirs();
		}
		return result;
	}

	/**
	 * Get application name
	 * 
	 * @return application name
	 */
	public String getAppName() {
		return getProperty(KEY_APP_NAME);
	}

	// -------------------------------------------------------------------------
	// --- SSL

	/**
	 * Get keystore for the application
	 * 
	 * @return keystore
	 */
	public String getKeyStore() {
		return getProperty(KEY_KEYSTORE);
	}

	/**
	 * Get keystore password
	 * 
	 * @return keystore password
	 */
	public String getKeyStorePasswd() {
		return getProperty(KEY_KEYSTORE_PASSWD);
	}

	/**
	 * Get truststore for the application
	 * 
	 * @return truststore for the application
	 */
	public String getTrustStore() {
		return getProperty(KEY_TRUSTSTORE);
	}

	/**
	 * Get truststore password
	 * 
	 * @return truststore password
	 */
	public String getTrustStorePasswd() {
		return getProperty(KEY_TRUSTSTORE_PASSWD);
	}

	// -------------------------------------------------------------------------
	// --- Vlab modules location's

	/**
	 * Get Communication Proxy Module
	 * 
	 * @return location of the commProxy module
	 */
	public String getCommProxyLocation() {
		return getProperty(KEY_COMM_PROXY_LOCATION);
	}

	/**
	 * Get Authentication Module Location
	 * 
	 * @return location of the Authentication Module
	 */
	public String getAuthModuleLocation() {
		return getProperty(KEY_AUTH_MODULE_LOCATION);
	}

	/**
	 * Get Monitoring Module Location
	 * 
	 * @return location of the Monitoring Module
	 */
	public String getMonitoringModuleLocation() {
		return getProperty(KEY_MON_MODULE_LOCATION);
	}

	/**
	 * Get Global Scheduler Module Location
	 * 
	 * @return location of the Global Scheduler Module
	 */
	public String getGlobalSchedulerModuleLocation() {
		return getProperty(KEY_GS_MODULE_LOCATION);
	}

	/**
	 * Get SMM Module Location
	 * 
	 * @return location of the SMM Module
	 */
	public String getSMMModuleLocation() {
		return getProperty(KEY_SMM_MODULE_LOCATION);
	}

	/**
	 * Get SZD Module Location
	 * 
	 * @return location of the SZD Module
	 */
	public String getSZDModuleLocation() {
		return getProperty(KEY_SZD_MODULE_LOCATION);
	}

	/**
	 * The custom property getter method by the given key name
	 * 
	 * @param key the name of the property key @ the VlabException is thrown
	 *        when the key is not found
	 */
	public String getProperty(String key) {
		String property = this.properties.getProperty(key);
		if (property != null) {
			property = property.trim();
		}
		return property;
	}

}
