/*
 * ConfigurationFactory.java
 *
 * Created on March 8, 2005, 4:04 PM
 */

package pl.psnc.vlab.conf;

import java.util.HashMap;
import java.util.Map;

import pl.psnc.vlab.exception.VlabException;

/**
 * Abstract factory for retrieving the configuration object. The factory has to
 * be initialised (see initConfig) before use.
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class ConfigFactory {

	/** Stores an instance of configuration */
	private static Map<String, IConfiguration> configMap = new HashMap<String, IConfiguration>();

	/** default configuration name */
	private static String configName;

	/**
	 * Initialise the Configuration with the given configuration implementation
	 * and the given configuration key.
	 * 
	 * @param configKey configuration key
	 * @param configuration configuration implementation
	 */
	public static void initConfig(String configKey, IConfiguration configuration) {
		configName = configKey;
		// add configuration to the pool
		configMap.put(configKey, configuration);
	}

	/**
	 * Get an instance of the configuration object. <B>NOTE !!!</B> Do not use
	 * this method with more then one module at the same time, which are running
	 * at the same java virtual machine.
	 * 
	 * @return configuration object
	 * @throws VlabException thrown when the configuration initialisation failed
	 */
	public static IConfiguration getConfigFactory() throws VlabException {
		return getConfigFactory(configName);
	}

	/**
	 * Get an instance of the configuration object
	 * 
	 * @param cfgName properties file with the configuration
	 * @return configuration object
	 * @throws VlabException thrown when the configuration initialisation failed
	 */
	@SuppressWarnings("unchecked")
	public static IConfiguration getConfigFactory(String cfgName) throws VlabException {
		IConfiguration configuration = configMap.get(cfgName);
		if (configuration == null) {
			configuration = new ConfigurationImpl(cfgName);
			configMap.put(cfgName, configuration);
		}
		return configuration;
	}

}
