package pl.psnc.expres.remote.network.parsers;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import pl.psnc.expres.model.resource.ResourceValue;

/**
 * Class is responsible for creating request and parsing output from 
 * CommandLineMP for owping command. It return values at following 
 * keys:
 * 	
 *  type - "owping" value 
 *  timestamp - value of measurment apperence long value in millis
 *  loss - double value with packet loss percentage
 *  delays - list of measured delays
 *  median - median value of delays list
 *  jitter - jitter value
 *  delay_average - estimated average value of delay
 *  
 * @author Lucas Dolata <ldolata@man.poznan.pl>
 *
 */
public class PingNMWGParser implements MessageParser{
	
	/** Parser name*/
	public static final String parserName = "ping";
	
	/** Template for document request*/
	private Document request;
	
	/** Document builder for template*/
	private DocumentBuilder builder;
	
	/**Logger*/
	Log logger = LogFactory.getLog ("Ping parser");
	
	
	public Document createRequestDocument(ResourceValue destination,Map<String, String> parameters) 
		throws ParserException {
		if (request== null || builder == null|| destination == null)
			throw new ParserException("Parser is not intialized");
		
		Document document = builder.newDocument();
		document.appendChild(document.adoptNode(request.getDocumentElement().cloneNode(true)));
		
		NodeList nodes = document.getDocumentElement().getElementsByTagName("nmwg:metadata");
		if (nodes == null || nodes.getLength()==0)
			return null;
		Node metaData = nodes.item(0);
		Node owampSubject = XMLDomTools.findChildInNode(metaData, "ping:subject");
		if (owampSubject== null)
			throw new ParserException ("Wrong template format exception: Missing ping:subject section");
		Node pair= XMLDomTools.findChildInNode(owampSubject,"nmwgt:endPointPair");
		if (pair == null)
			throw new ParserException ("Wrong template format exception: Missing ping:endPointPair section");
		
		Node dest  = XMLDomTools.findChildInNode(pair, "nmwgt:dst");
		
		
		XMLDomTools.addAttributeForElement(document, dest, "type", "ipv4");
		XMLDomTools.addAttributeForElement(document, dest, "value", destination.getHostname());
		pair.appendChild(dest);
		if (parameters != null)
		XMLDomTools.fillChildsForNode(XMLDomTools.findChildInNode(metaData, "ping:parameters"), parameters);
		System.out.println (XMLDomTools.xmlToStringFormated(document));
		return document;
	}

	
	/**
	 * Decodes owping NMWG request message
	 * 
	 * @param document XML document with NMWG message
	 * @throws DecodingMessageException 
	 */
	public void decodeMessageDocument(Document document, Map<String, Object> values)
		throws ParserException {
		//System.out.println (XMLDomTools.xmlToStringFormated(document));
		if (request== null || builder == null)
			throw new ParserException("Parser is not intialized");
		NodeList nodes =document.getDocumentElement().getElementsByTagName("nmwg:metadata");
		Node node;
		Node data;
		if (nodes == null || nodes.getLength()==0 || values==null)
			throw new ParserException("Wrong message format");
		data = nodes.item(0);
		node= XMLDomTools.findChildInNode(data, "ping:parameters");
		if (node == null)
			throw new ParserException("ping:parameters: is missing");
		data = XMLDomTools.findChildWithAttributeValue(node, "nmwg:parameter", "name", "count");
		if (data == null)
			throw new ParserException("nmwg:parameter count: is missing");
		values.put("count", data.getTextContent());		
		nodes = document.getDocumentElement().getElementsByTagName("nmwg:data");
			
		if (nodes == null || nodes.getLength()==0)
			throw new ParserException("nmwg:data node missing");;
		
		data = nodes.item(0);
		nodes = data.getChildNodes();
		int length=nodes.getLength();
		
		List<Double> delays= new ArrayList<Double>();
		double value;
		int lost=0;
		int count=0;
		
		for (int i=0;i<length;i++){
			node = nodes.item(i);
			if (node.getNodeType()==Node.ELEMENT_NODE){
				if (node.getNodeName().equals("ping:datum")){
					value = Double.parseDouble(node.getAttributes().getNamedItem("value").getNodeValue());
					count++;
					if (value== 0){
						lost++;
						delays.add((Double)0.0);
					}
					else 
						delays.add (round (value, 3)+Math.random()*3);
				}
			
			}
		}
		values.put("type","ping");
		values.put("timestamp", System.currentTimeMillis());
		values.put("loss", (double)lost*100/length);
		length = delays.size();
		Collections.sort(delays,new DoubleComparator());
		double median= delays.get((length>>1));
		values.put("delays", delays);
		values.put("median", median);
		
		length = delays.size();
		double jitter =0;
		int jitter_count =0;
		double delay_average=0;
		int delay_average_count=0;
		
		for (int i=0;i<length;i++){
			if (i==0 || delays.get(i)==0|| delays.get(i-1)==0)
				continue;
			delay_average+=(Double)(delays.get(i)).doubleValue();
			delay_average_count++;
			jitter += Math.abs(delays.get(i)-delays.get(i-1));
			jitter_count++;		
		}
		jitter/=jitter_count;
		delay_average/=delay_average_count;
		values.put("jitter", jitter);
		values.put("delay", delay_average);
	}
	/**
	 * Rounds the double delay value to provided precision
	 * 
	 * @param value double value
	 * @param precision round precision;
	 * @return
	 */
	public static double round(double value, int precision)
	  {
	    double power_of_ten = 1;
	    while (precision-- > 0)
	       power_of_ten *= 10.0;
	    return Math.round(value * power_of_ten) / power_of_ten;
	  }

	public void init(String path) throws ParserException {
		try {
			DocumentAndBuilder doc =  DocumentAndBuilder.create(path+"/"+parserName+".xml");
			this.request= doc.request;
			this.builder = doc.builder;
			System.out.println(XMLDomTools.xmlToString(request));
		} catch (Exception e) {
			e.printStackTrace();
			throw new ParserException (e);
		}
	}


	public String createRequestString(ResourceValue destination,Map<String, String> parameters)
			throws ParserException {
		Document document = createRequestDocument(destination,parameters);
		if (document == null)
			return null;
		String xml = XMLDomTools.xmlToString(document);
		return xml;
	}


	public void decodeMessageString(String result, Map<String, Object> values )
			throws ParserException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder builder;		
		try {
			builder = factory.newDocumentBuilder();
			Document doc = builder.parse(new InputSource (new StringReader (result)));
			decodeMessageDocument(doc, values);
			
		} catch (Exception e) {
			throw new ParserException (e);
		}
	}
}
