/**
 * ResourceTypeFactory.java
 * Created on Nov 8, 2007, 3:42:13 PM
 */
package pl.psnc.expres.model.factory;

import java.io.FileInputStream;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import pl.psnc.expres.model.factory.helper.ModelKeys;
import pl.psnc.expres.model.type.ResourceType;
import pl.psnc.expres.model.type.ResourceTypeMapper;
import pl.psnc.expres.model.util.JsonReaderHelper;
import pl.psnc.vlab.exception.ResourceNotFoundException;

/**
 * {@link ResourceTypeFactory} class - the Type Factory is used to get the list
 * of resource types. The type component describes the resource type name, id
 * and icons
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 */
public class ResourceTypeFactory {

	/** Logger instance */
	private static Log log = LogFactory.getLog(ResourceTypeFactory.class);

	/** Stores the list of resource types */
	private static List<ResourceType> types;

	/**
	 * Get the resource type by the given id. Resource types are loaded from
	 * default location - ModelKeys.Type_LOCATION
	 * 
	 * @param id type identifier
	 * @return resource type by the given id
	 * @throws ResourceNotFoundException thrown when resource type cannot be
	 *         found
	 */
	public static ResourceType getResourceType(int id) throws ResourceNotFoundException {
		return getResourceType(id, ModelKeys.TYPE_LOCATION);
	}

	/**
	 * Get the resource type by the given id
	 * 
	 * @param id type identifier
	 * @param filePath path to the source file
	 * @return type for the given id or {@link ResourceNotFoundException}
	 * @throws ResourceNotFoundException thrown when resource type cannot be
	 *         found
	 */
	public static ResourceType getResourceType(int id, String filePath)
			throws ResourceNotFoundException {
		if (types == null) {
			types = getTypes(filePath);
		}
		for (ResourceType resourceType : types) {
			if (resourceType.getId() == id) {
				return resourceType;
			}
		}
		throw new ResourceNotFoundException("Resource type [type id=" + id
				+ "] has not been found.");
	}

	/**
	 * Load a resource types from the given source
	 * 
	 * @param filePath path to the resource file
	 * @return resource types from the given source
	 * @throws ResourceNotFoundException thrown when method fails
	 */
	public static List<ResourceType> getTypes(String filePath) throws ResourceNotFoundException {
		try {
			InputStream inStream = new FileInputStream(filePath);
			return getTypes(inStream);
		} catch (Exception e) {
			throw new ResourceNotFoundException(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * Load a resource types from the given source
	 * 
	 * @param filePath path to the resource file
	 * @return resource types from the given source
	 * @throws ResourceNotFoundException thrown when method fails
	 */
	public static List<ResourceType> getTypes(InputStream inStream)
			throws ResourceNotFoundException {
		if (types == null) {
			Object res = JsonReaderHelper.load(inStream, ResourceTypeMapper.class);
			if (res == null) {
				log.error("ResourceTypes are NULL");
				return null;
			}
			List<ResourceType> typeList = Arrays.asList(((ResourceTypeMapper) res).getType());
			types = new ArrayList<ResourceType>();
			for (ResourceType rType : typeList) {
				types.add(getResourceType(rType));
			}
		}
		return types;
	}

	// -------------------------------------------------------------------
	// --------- Private helper methods

	/**
	 * Get Resource Type based on the given ResourceType object
	 * 
	 * @param type ResourceType type of the given resource
	 * @return ResourceType object wrapped with concrete implementation of
	 *         AbstractResourceType
	 * @throws ResourceNotFoundException
	 */
	@SuppressWarnings("unchecked")
	private static ResourceType getResourceType(ResourceType type) throws ResourceNotFoundException {
		try {
			Class className = Class.forName(type.getClassName());
			Constructor<ResourceType> constructor = className
					.getConstructor(new Class[] { ResourceType.class });
			return constructor.newInstance(new Object[] { type });
		} catch (Exception e) {
			throw new ResourceNotFoundException(e.getMessage(), e);
		}
	}

}
