/*
 * SSABundleFactory.java
 * 
 */

package pl.psnc.expres.wfm.util.i18n.helper;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import pl.psnc.vlab.util.i18n.ResourceBundleManager;

/**
 * {@link BundleFactory} class - Factory class for the properties retrieval. The
 * GUI factory deals with all the properties from GUI forms
 * 
 * @author <a href="mailto:osa@man.poznan.pl">Dominik Stoklosa (~osa~)</a>
 * @email osa@man.poznan.pl
 * 
 */
public abstract class BundleFactory {

	/** The resource property file for the GUI forms */
	private static final String BUNDLE_GUI = "bundle/gui";

	/** The resource property file for the massages shown by the application */
	private static final String BUNDLE_MESSAGES = "bundle/messages";

	/**
	 * Stores the key with message sent when the specified key value cannot be
	 * found
	 */
	private static final String KEY_NOT_FOUND = "message.key.not.found";

	/** Holds instances of resource bundle managers */
	private static Map<String, ResourceBundleManager> bundleMap;

	/**
	 * Get the value for the given key for corresponding resource bundle: see
	 * BundleFactory.BUNDLE_GUI
	 * 
	 * @param key key for the desired string
	 * @return value of the given key
	 */
	public static String getGuiValue(String key) {
		return getBundleManager(BUNDLE_GUI).getValue(key);
	}

	/**
	 * Get the formatted value for the given key for corresponding resource
	 * bundle: see BundleFactory.BUNDLE_MESSAGES
	 * 
	 * @param key key for the desired string
	 * @param values list of values for the formated key
	 * @return value of the given key
	 */
	public static String getGuiValue(String key, String... values) {
		return getBundleManager(BUNDLE_GUI).getFormattedValue(key, values);
	}

	/**
	 * Get the value for the given key for corresponding resource bundle: see
	 * BundleFactory.BUNDLE_MESSAGES
	 * 
	 * @param key key for the desired string
	 * @return value of the given key
	 */
	public static String getMessageValue(String key) {
		return getBundleManager(BUNDLE_MESSAGES).getValue(key);
	}

	/**
	 * Get the formatted value for the given key for corresponding resource
	 * bundle: see BundleFactory.BUNDLE_MESSAGES
	 * 
	 * @param key key for the desired string
	 * @param values list of values for the formated key
	 * @return value of the given key
	 */
	public static String getMessageValue(String key, String... values) {
		return getBundleManager(BUNDLE_MESSAGES).getFormattedValue(key, values);
	}

	// -----------------------------------------------------------------------
	// ---- Private helper methods

	/**
	 * Gets the ResourceBundleManager for the given bundle
	 * 
	 * @param bundle bundle name
	 * @return new ResourceBundleManager for the given bundle
	 */
	private static ResourceBundleManager getBundleManager(String bundle) {
		if (bundleMap == null) {
			bundleMap = new HashMap<String, ResourceBundleManager>();
		}
		ResourceBundleManager rb = storeBundleManager(bundle);
		return rb;
	}

	/**
	 * Helper methods used to create new ResourceBundleManager for the given
	 * bundle
	 * 
	 * @param bundle bundle name
	 * @return new ResourceBundleManager for the given bundle
	 */
	private static ResourceBundleManager storeBundleManager(String bundle) {
		ResourceBundleManager rb = bundleMap.get(bundle);
		if (rb == null) {
			rb = new ResourceBundleManager(bundle, Locale.getDefault(), KEY_NOT_FOUND);
			bundleMap.put(bundle, rb);
		}
		return rb;
	}

}
